(function($) {
    'use strict';

    $(function() {
        // Keep system notices in a dedicated row above the header.
        function relocateSystemNotices() {
            var $adminRoot = $('#i10n-push-subscriber-admin');
            if (!$adminRoot.length) return;
            var $noticeSlot = $('#i10n-push-system-notices');
            if (!$noticeSlot.length) return;
            $adminRoot.find('.i10n-push-top-row1 .notice, > .notice').each(function() {
                $noticeSlot.append($(this));
            });
        }

        relocateSystemNotices();

        if (window.MutationObserver) {
            var observer = new MutationObserver(function(mutations) {
                var shouldRelocate = false;
                mutations.forEach(function(mutation) {
                    if (mutation.addedNodes && mutation.addedNodes.length) {
                        shouldRelocate = true;
                    }
                });
                if (shouldRelocate) {
                    relocateSystemNotices();
                }
            });
            var adminRoot = document.getElementById('i10n-push-subscriber-admin');
            if (adminRoot) {
                observer.observe(adminRoot, { childList: true, subtree: true });
            }
        }

        // --- Global Variables & Functions ---
        var pollingInterval;
        var pollingTimeout;

        function stopPolling() {
            clearInterval(pollingInterval);
            clearTimeout(pollingTimeout);
        }

        function showNotice(selector, message, type) {
            var $noticeContainer = $(selector);
            $noticeContainer.removeClass('notice-info notice-success notice-warning notice-error').addClass('notice-' + type);
            if ($noticeContainer.find('p').length === 0) {
                $noticeContainer.append('<p></p>');
            }
            var $paragraph = $noticeContainer.find('p').first();
            var safeMessage = String(message || '').replace(/<br\s*\/?>/gi, '\n');
            var parts = safeMessage.split('\n');
            var $strong = $('<strong></strong>');
            $paragraph.empty().append($strong);
            parts.forEach(function(part, index) {
                if (index > 0) {
                    $strong.append('<br>');
                }
                $strong.append(document.createTextNode(part));
            });
        }

        // --- Polling Logic ---
        function pollForApiKey() {
            if (!i10n_push_data || !i10n_push_data.ajax_url) {
                console.error('[i10n-push] ajax_url missing, cannot poll verification.');
                return;
            }
            console.log('Polling for API key...', i10n_push_data.site_url);
            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_check_verification',
                _wpnonce: i10n_push_data.nonce, // Use a general nonce
                site_url: i10n_push_data.site_url
            }).done(function(response) {
                if (response.success && (response.data.status === 'verified' || response.data.status === 'connected' || response.data.api_key)) {
                    stopPolling();
                    showNotice('.i10n-push-notice-container', '???????????...', 'success');

                    // Now, save the API key
                    $.post(i10n_push_data.ajax_url, {
                        action: 'i10n_push_save_api_key',
                        _wpnonce: i10n_push_data.nonce,
                        api_key: response.data.api_key
                    }).done(function(saveResponse) {
                        if (saveResponse.success) {
                            showNotice('.i10n-push-notice-container', '???????????????', 'success');
                            setTimeout(function() { window.location.reload(); }, 1500);
                        } else {
                            showNotice('.i10n-push-notice-container', '???? API ???????????????', 'error');
                        }
                    }).fail(function() {
                        showNotice('.i10n-push-notice-container', '?? API ???????????', 'error');
                    });
                } else {
                    console.log('Polling response', response);
                }
            }).fail(function(xhr, status) {
                console.error('Polling request failed', status, xhr);
            });
        }

        function startPolling(noticeSelector) {
            stopPolling(); // Ensure no multiple intervals are running
            pollingInterval = setInterval(pollForApiKey, 5000); // Poll every 5 seconds
            pollingTimeout = setTimeout(function() {
                stopPolling();
                showNotice(noticeSelector, '連線已逾時。<br>我們已寄出驗證信至您的信箱，請點擊信中連結完成啟用。如果您沒有收到郵件，請檢查垃圾郵件匣，或稍後再試一次。', 'warning');
            }, 300000); // Stop polling after 5 minutes
        }

        // --- Connect Button Handler (for welcome notice) ---
        $('body').on('click', '#i10n-push-connect-button', function(e) {
            e.preventDefault();
            var $button = $(this);
            var originalText = $button.text();
            var $notice = $button.closest('.notice');
            if (!$notice.length) {
                $notice = $('.i10n-push-admin-notices');
                $notice.addClass('notice i10n-push-notice-container');
            }
            $button.prop('disabled', true).text('處理中...');

            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_connect',
                _wpnonce: i10n_push_data.nonce
            }).done(function(response) {
                if (response.success) {
                    if (response.data.status === 'pending') {
                        $notice.addClass('i10n-push-notice-container'); // Add a class for targeting
                        showNotice($notice, response.data.message, 'info');
                        $button.remove();
                        startPolling($notice);
                    } else if (response.data.status === 'connected') {
                        showNotice($notice, response.data.message, 'success');
                        setTimeout(function() { window.location.reload(); }, 1500);
                    }
                } else {
                    $button.prop('disabled', false).text(originalText);
                    showNotice($notice, response.data.message || '發生錯誤，請稍後再試。', 'error');
                }
            }).fail(function() {
                $button.prop('disabled', false).text(originalText);
                showNotice($notice, '發生未知的錯誤，請稍後再試。', 'error');
            });
        });

        // --- TOS gating (before connect) ---
        var $tosLink = $('#i10n-push-tos-link');
        var $checkTosButton = $('#i10n-push-check-tos-button');
        var $tosStatus = $('#i10n-push-tos-status');
        var $connectBtn = $('#i10n-push-connect-button');

        function setTosStatus(text, type) {
            if (!$tosStatus.length) return;
            var color = (type === 'ok') ? '#1d7f35' : (type === 'warn') ? '#b45309' : '#6b7280';
            $tosStatus.text(text).css('color', color);
        }

        function checkTosConsent(onDone) {
            if (!i10n_push_data || !i10n_push_data.ajax_url) return;
            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_check_tos_consent',
                nonce: i10n_push_data.nonce
            }).done(function (resp) {
                var consented = !!(resp && resp.success && resp.data && resp.data.consented);
                if (consented) {
                    setTosStatus('已同意服務條款', 'ok');
                    if ($connectBtn.length) {
                        $connectBtn.prop('disabled', false);
                    }
                } else {
                    setTosStatus('尚未同意服務條款', 'warn');
                }
                if (typeof onDone === 'function') onDone(consented);
            }).fail(function () {
                setTosStatus('無法檢查同意狀態', 'warn');
                if (typeof onDone === 'function') onDone(false);
            });
        }

        if ($tosLink.length && $connectBtn.length) {
            checkTosConsent();

            $tosLink.on('click', function () {
                setTosStatus('請在新分頁完成同意後回來。', 'info');
                var attempts = 0;
                var maxAttempts = 30; // ~60s
                var timer = setInterval(function () {
                    attempts++;
                    checkTosConsent(function (consented) {
                        if (consented || attempts >= maxAttempts) {
                            clearInterval(timer);
                        }
                    });
                }, 2000);
            });

            $checkTosButton.on('click', function (e) {
                e.preventDefault();
                checkTosConsent();
            });
        }

        // --- Settings Page Logic ---
        var $adminPage = $('#i10n-push-subscriber-admin');
        if (!$adminPage.length) {
            return; // Exit if not on the settings page
        }

        // Manual "check verification" button (pending state)
        $('body').on('click', '#i10n-push-check-verification-button', function(e) {
            e.preventDefault();
            var $btn = $(this);
            var $noticeContainer = $('.i10n-push-admin-notices');
            $noticeContainer.addClass('i10n-push-notice-container notice notice-info');
            $btn.prop('disabled', true).text('檢查中...');
            // Trigger an immediate poll, then start interval polling
            pollForApiKey();
            startPolling($noticeContainer);
            setTimeout(function() { $btn.prop('disabled', false).text('再次檢查驗證'); }, 5000);
        });

        // Check if the settings page is in a pending state on load
        if ($adminPage.find('.i10n-push-status-pending').length > 0) {
            var $noticeContainer = $('.i10n-push-admin-notices');
            $noticeContainer.addClass('i10n-push-notice-container notice notice-info');
            showNotice($noticeContainer, '等待電子郵件驗證中... 此頁面將在驗證成功後自動更新。', 'info');
            startPolling($noticeContainer);
        }

        // --- Subscription Management Logic ---
        var $subscriptionCheckboxes = $('.subscription-checkbox');
        var $summaryContainer = $('#i10n-push-summary');
        var $subtotalSpan = $('#i10n-push-subtotal');
        var $discountDesignatedSpan = $('#i10n-push-discount-designated');
        var $discountUpgradeSpan = $('#i10n-push-discount-upgrade');
        var $discountManagedSpan = $('#i10n-push-discount-managed');
        var $discountEarlyBirdSpan = $('#i10n-push-discount-earlybird');
        var $earlyBirdLabel = $('#i10n-push-earlybird-label');
        var $totalSpan = $('#i10n-push-total');
        var $saveButton = $('#save-subscriptions-button');
        var $quoteButton = $('#request-quote-button');
        var $reapplyButton = $('#i10n-push-reapply-button');
        var managedFreeCredit = 0;
        var tier3Price = 200;
        var earlyBirdEligible = true;
        var prorationRate = 1.0;
        var prorationDaysRemaining = null;
        var designatedFreeFirstYearEligible = true;
        if (i10n_push_data) {
            if (typeof i10n_push_data.managed_free_credit !== 'undefined') {
                managedFreeCredit = parseFloat(i10n_push_data.managed_free_credit) || 0;
            } else if (i10n_push_data.available_items && typeof i10n_push_data.available_items.managed_free_credit !== 'undefined') {
                managedFreeCredit = parseFloat(i10n_push_data.available_items.managed_free_credit) || 0;
            }
            if (i10n_push_data.available_items && typeof i10n_push_data.available_items.tier3_price !== 'undefined') {
                tier3Price = parseFloat(i10n_push_data.available_items.tier3_price) || tier3Price;
            }
            if (i10n_push_data.available_items && typeof i10n_push_data.available_items.early_bird_eligible !== 'undefined') {
                earlyBirdEligible = !!i10n_push_data.available_items.early_bird_eligible;
            }
            if (i10n_push_data.available_items && typeof i10n_push_data.available_items.designated_free_first_year_eligible !== 'undefined') {
                designatedFreeFirstYearEligible = !!i10n_push_data.available_items.designated_free_first_year_eligible;
            }
            if (i10n_push_data.available_items && typeof i10n_push_data.available_items.proration_rate !== 'undefined') {
                prorationRate = parseFloat(i10n_push_data.available_items.proration_rate) || prorationRate;
            }
            if (i10n_push_data.available_items && typeof i10n_push_data.available_items.proration_days_remaining !== 'undefined') {
                prorationDaysRemaining = parseInt(i10n_push_data.available_items.proration_days_remaining, 10);
                if (isNaN(prorationDaysRemaining)) {
                    prorationDaysRemaining = null;
                }
            }
        }

        var allItems = [];
        var itemsByDomain = {};
        if (i10n_push_data.available_items) {
            allItems = [].concat(i10n_push_data.available_items.plugins || [], i10n_push_data.available_items.themes || []);
            allItems.forEach(function (item) {
                if (item && item.text_domain) {
                    itemsByDomain[item.text_domain] = item;
                }
            });
        }

        // Precompute existing designated-free subscriptions to know remaining free slots
        var existingDesignatedFree = 0;
        if (designatedFreeFirstYearEligible && allItems.length > 0) {
            existingDesignatedFree = allItems.filter(function (item) {
                return item.is_subscribed && item.source === 'designated-free';
            }).length;
        }

        // Helper: get early bird rate based on UTC+8 deadlines
        function getEarlyBirdRate() {
            var now = new Date();
            var nowMs = now.getTime();
            var offsetMs = 8 * 60 * 60 * 1000;
            var nowUtc8 = new Date(nowMs + (now.getTimezoneOffset() * 60 * 1000) + offsetMs);
            var superEarly = Date.parse('2026-01-31T23:59:59+08:00');
            var veryEarly = Date.parse('2026-03-31T23:59:59+08:00');
            var early = Date.parse('2026-05-31T23:59:59+08:00');
            var ts = nowUtc8.getTime();
            if (ts <= superEarly) {
                return { rate: 0.3, label: '(超早鳥 3折)' };
            } else if (ts <= veryEarly) {
                return { rate: 0.5, label: '(早早鳥 5折)' };
            } else if (ts <= early) {
                return { rate: 0.7, label: '(早鳥 7折)' };
            }
            return { rate: 1.0, label: '' };
        }

        function updateSubscriptionSummary() {
            var subtotalOriginal = 0;
            var quoteRequiredCount = 0;
            var subscribableCount = 0;
            var designatedFreeLimit = designatedFreeFirstYearEligible ? Math.max(0, 2 - existingDesignatedFree) : 0;
            var designatedFreeSelectedPrices = [];
            var selectedPaidPrices = [];
            var upgradeDiscount = 0;

            var $currentSelections = $subscriptionCheckboxes.filter(':checked').filter(function () {
                return !$(this).is(':disabled');
            });

            function round0(n) { return Math.round(n); }

            var selectedTextDomains = [];
            $currentSelections.each(function() {
                var $this = $(this);
                var annualPrice = parseInt($this.data('price') || 0);
                var proratedPrice = round0(annualPrice * prorationRate);
                var tier = $this.data('tier');
                var source = $this.data('source');
                var textDomain = $this.val();
                if (textDomain) {
                    selectedTextDomains.push(textDomain);
                }

                if (tier === 'quote_required') {
                    quoteRequiredCount++;
                } else if (annualPrice > 0) {
                    subscribableCount++;
                    if (source === 'designated-free') {
                        if (designatedFreeFirstYearEligible) {
                            designatedFreeSelectedPrices.push(proratedPrice);
                        } else {
                            selectedPaidPrices.push(proratedPrice);
                        }
                    } else {
                        selectedPaidPrices.push(proratedPrice);
                    }
                }
            });

            var dependencySeen = {};
            var remainingDays = prorationDaysRemaining !== null ? prorationDaysRemaining : Math.round(prorationRate * 365);
            selectedTextDomains.forEach(function (textDomain) {
                var itemData = itemsByDomain[textDomain];
                if (!itemData || !itemData.requires || !itemData.requires.length) {
                    return;
                }
                itemData.requires.forEach(function (dependency) {
                    if (!dependency || dependencySeen[dependency]) {
                        return;
                    }
                    var depItem = itemsByDomain[dependency];
                    if (!depItem || !depItem.is_subscribed) {
                        return;
                    }
                    var paidPrice = parseInt(itemData.price || 0, 10);
                    var depPrice = parseInt(depItem.price || 0, 10);
                    var priceDiff = Math.max(0, paidPrice - depPrice);
                    var credit = round0(priceDiff * (remainingDays / 365));
                    if (credit > 0) {
                        upgradeDiscount += credit;
                    }
                    dependencySeen[dependency] = true;
                });
            });

            designatedFreeSelectedPrices.sort(function(a, b) { return b - a; });
            var designatedDiscount = 0;
            designatedFreeSelectedPrices.forEach(function(price, idx) {
                if (idx < designatedFreeLimit) {
                    // First year: make designated-free items free for this term (prorated to renewal date)
                    designatedDiscount += round0(price);
                } else {
                    selectedPaidPrices.push(round0(price));
                }
            });

            var subtotalAfterDesignated = selectedPaidPrices.reduce(function(acc, v) { return acc + v; }, 0);
            subtotalOriginal = subtotalAfterDesignated + designatedDiscount;

            var earlyInfo = earlyBirdEligible ? getEarlyBirdRate() : { rate: 1.0, label: '' };
            var subtotalAfterUpgrade = Math.max(0, subtotalAfterDesignated - upgradeDiscount);
            var managedDiscount = Math.min(managedFreeCredit, subtotalAfterUpgrade);
            var subtotalAfterManaged = subtotalAfterUpgrade - managedDiscount;
            var earlyBirdDiscount = round0(subtotalAfterManaged * (1 - earlyInfo.rate));
            var finalTotal = round0(subtotalAfterManaged * earlyInfo.rate);

            // Update UI
            $subtotalSpan.text('NT$ ' + subtotalOriginal);
            $discountDesignatedSpan.text('NT$ ' + designatedDiscount);
            $discountUpgradeSpan.text('NT$ ' + upgradeDiscount);
            $discountManagedSpan.text('NT$ ' + round0(managedDiscount));
            $discountEarlyBirdSpan.text('NT$ ' + earlyBirdDiscount);
            $earlyBirdLabel.text(earlyInfo.label);
            $totalSpan.text('NT$ ' + finalTotal);

            var hasSelections = $currentSelections.length > 0;
            var hasPricedSelections = subscribableCount > 0;

            if (hasPricedSelections) {
                $summaryContainer.show();
            } else {
                $summaryContainer.hide();
            }

            if (quoteRequiredCount > 0 && hasPricedSelections) {
                $quoteButton.show();
                $saveButton.show();
            } else if (quoteRequiredCount > 0) {
                $quoteButton.show();
                $saveButton.hide();
            } else if (hasPricedSelections) {
                $quoteButton.hide();
                $saveButton.show();
            } else {
                $quoteButton.hide();
                $saveButton.hide();
            }
        }

        // Bind change event to all subscription checkboxes
        $subscriptionCheckboxes.on('change', updateSubscriptionSummary);

        // Bind change event to "Select All" checkboxes
        $('#select-all-plugins-checkbox').on('change', function() {
            var isChecked = $(this).is(':checked');
            $('table:has(#select-all-plugins-checkbox) .subscription-checkbox').each(function() {
                if (!$(this).is(':disabled')) {
                    $(this).prop('checked', isChecked);
                }
            });
            updateSubscriptionSummary();
        });

        $('#select-all-themes-checkbox').on('change', function() {
            var isChecked = $(this).is(':checked');
            $('table:has(#select-all-themes-checkbox) .subscription-checkbox').each(function() {
                if (!$(this).is(':disabled')) {
                    $(this).prop('checked', isChecked);
                }
            });
            updateSubscriptionSummary();
        });

        // Handle Save Subscriptions Button Click
        $saveButton.on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            $button.prop('disabled', true).text('處理中...');

            var selectedSubscriptions = [];
            var translationFiles = {}; // Re-implement the logic to collect file URLs

            $subscriptionCheckboxes.each(function() {
                var $this = $(this);
                if ($this.is(':checked') && !$this.is(':disabled') && $this.data('tier') !== 'quote_required') {
                    var textDomain = $this.val();
                    selectedSubscriptions.push(textDomain);
                    
                    var poFileUrl = $this.data('po-file-url');
                    if (poFileUrl) {
                        translationFiles[textDomain] = poFileUrl;
                    }
                }
            });

            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_create_order',
                nonce: i10n_push_data.nonce,
                site_url: i10n_push_data.site_url,
                subscriptions: selectedSubscriptions,
                translation_files: translationFiles, // Pass the collected file URLs
                active_items: i10n_push_data.active_items || []
            }).done(function(response) {
                if (response.success) {
                    alert(response.data.message || '訂單已建立，請至您的信箱查收付款資訊。');
                    window.location.reload();
                } else {
                    alert(response.data.message || '更新訂閱失敗。');
                    $button.prop('disabled', false).text('訂閱選取項目');
                }
            }).fail(function() {
                alert('發生未知的錯誤，請稍後再試。');
                $button.prop('disabled', false).text('訂閱選取項目');
            });
        });

        // Handle Request Quote Button Click
        $quoteButton.on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            $button.prop('disabled', true).text('處理中...');

            var quoteItems = [];
            $subscriptionCheckboxes.each(function() {
                var $this = $(this);
                if ($this.is(':checked') && !$this.is(':disabled') && $this.data('tier') === 'quote_required') {
                    quoteItems.push($this.val());
                }
            });

            if (quoteItems.length === 0) {
                alert('請選擇需要報價的項目。');
                $button.prop('disabled', false).text('請求報價');
                return;
            }

            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_request_quote',
                nonce: i10n_push_data.nonce,
                site_url: i10n_push_data.site_url,
                text_domains: quoteItems,
                active_items: i10n_push_data.active_items || []
            }).done(function(response) {
                if (response.success) {
                    alert(response.data.message || '報價請求已送出。');
                    window.location.reload();
                } else {
                    alert(response.data.message || '請求報價失敗。');
                    $button.prop('disabled', false).text('請求報價');
                }
            }).fail(function() {
                alert('發生未知的錯誤，請稍後再試。');
                $button.prop('disabled', false).text('請求報價');
            });
        });

        // Handle Force Refresh Button Click
        $('#force-refresh-button').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            $button.prop('disabled', true).text('更新中...');

            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_force_refresh',
                nonce: i10n_push_data.nonce
            }).done(function(response) {
                if (response.success) {
                    // Redirect to the same page with a success message
                    window.location.href = 'options-general.php?page=' + i10n_push_data.plugin_name + '&message=refreshed';
                } else {
                    alert(response.data.message || '更新失敗。');
                    $button.prop('disabled', false).text('強制更新清單');
                }
            }).fail(function() {
                alert('發生未知的錯誤，請稍後再試。');
                $button.prop('disabled', false).text('強制更新清單');
            });
        });

        // Handle Force Update Translations Button Click
        $('#force-update-translations-button').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            $button.prop('disabled', true).text('處理中...');

            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_force_update_translations',
                nonce: i10n_push_data.nonce
            }).done(function(response) {
                if (response.success) {
                    alert(response.data.message || '強制更新翻譯已觸發。請稍候，翻譯檔案將在背景下載並安裝。');
                    window.location.reload();
                } else {
                    alert(response.data.message || '強制更新翻譯失敗。');
                    $button.prop('disabled', false).text('強制更新翻譯');
                }
            }).fail(function() {
                alert('發生未知的錯誤，請稍後再試。');
                $button.prop('disabled', false).text('強制更新翻譯');
            });
        });

        // Initial summary update on page load
        updateSubscriptionSummary();

        // Reapply flow for revoked sites
        $reapplyButton.on('click', function () {
            if (!confirm('確定要提交重新申請？審核通過後才會寄送驗證信。')) {
                return;
            }
            var reason = prompt('請填寫重新申請的原因（選填）：', '');
            $.post(i10n_push_data.ajax_url, {
                action: 'i10n_push_reapply_site',
                nonce: i10n_push_data.nonce,
                reason: reason || ''
            }, function (response) {
                if (response.success) {
                    alert(response.data.message || '已提交重新申請，請等待審核。');
                    $reapplyButton.prop('disabled', true);
                } else {
                    alert(response.data && response.data.message ? response.data.message : '提交失敗，請稍後再試。');
                }
            }).fail(function () {
                alert('提交失敗，請稍後再試。');
            });
        });
    });
})(jQuery);
