<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


/**
 * Handles the plugin self-update mechanism.
 *
 * @since      1.0.0
 * @package    i10n-push-subscriber
 * @subpackage i10n-push-subscriber/includes
 */
class I10n_Push_Plugin_Updater {

	/**
	 * The plugin slug.
	 *
	 * @var string
	 */
	protected $plugin_slug;

	/**
	 * The plugin version.
	 *
	 * @var string
	 */
	protected $version;

	/**
	 * The API Key for the connected site.
	 *
	 * @var string
	 */
	protected $api_key;

	/**
	 * The API endpoint for checking updates.
	 *
	 * @var string
	 */
	protected $api_url;

    /**
     * Cache key for update info.
     * 
     * @var string
     */
    protected $cache_key;

    /**
     * Host used to validate download/package URLs.
     *
     * @var string|null
     */
    protected $api_host;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_slug The plugin slug.
	 * @param string $version     The current plugin version.
	 */
	public function __construct( $plugin_slug, $version ) {
		$this->plugin_slug = $plugin_slug;
		$this->version     = $version;
		$this->api_key     = get_option( 'i10n_push_api_key', '' );
		$this->api_url     = I10N_PUSH_API_URL . '/check-plugin-update';
        $this->cache_key   = 'i10n_push_plugin_update_' . $this->plugin_slug;
        $this->api_host    = wp_parse_url( I10N_PUSH_API_URL, PHP_URL_HOST );

		// Hook into the transient update check
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_update' ) );

		// Hook into the plugin details popup
		add_filter( 'plugins_api', array( $this, 'check_info' ), 10, 3 );
	}

	/**
	 * Check for updates against the custom API.
	 *
	 * @param object $transient The transient object containing update info.
	 * @return object The modified transient object.
	 */
	public function check_update( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		// Get the remote version
		$remote_version = $this->request_update_check();

		if ( $this->is_update_available( $remote_version ) ) {
            $res = $this->map_remote_update_data( $remote_version );
			$transient->response[ $res->plugin ] = $res;
		}

		return $transient;
	}

	/**
	 * Get plugin information for the details popup.
	 *
	 * @param false|object|array $result The result object or false.
	 * @param string             $action The API action being performed.
	 * @param object             $args   Plugin API arguments.
	 * @return false|object|array The result object or false.
	 */
	public function check_info( $result, $action, $args ) {
		// Only handle our plugin
		if ( 'plugin_information' !== $action ) {
			return $result;
		}

		if ( empty( $args->slug ) || $this->plugin_slug !== $args->slug ) {
			return $result;
		}

		$remote_version = $this->request_update_check();

		if ( $remote_version ) {
			return $this->map_remote_info_data( $remote_version );
		}

		return $result;
	}

    /**
     * Check if a valid update is available from the remote version object.
     *
     * @param object|bool $remote_version The remote version object.
     * @return bool True if update is available, false otherwise.
     */
    protected function is_update_available( $remote_version ) {
        return (
            $remote_version 
            && isset( $remote_version->new_version ) 
            && version_compare( $this->version, $remote_version->new_version, '<' )
        );
    }

    /**
     * Map remote version data to WordPress update object format.
     *
     * @param object $remote_version The remote version object.
     * @return stdClass The mapped update object.
     */
    protected function map_remote_update_data( $remote_version ) {
        $res = new stdClass();
        $res->slug = $this->plugin_slug;
        $res->plugin = isset($remote_version->plugin) ? sanitize_text_field($remote_version->plugin) : $this->plugin_slug . '/' . $this->plugin_slug . '.php';
        $res->new_version = sanitize_text_field($remote_version->new_version);
        $res->url = isset($remote_version->url) ? esc_url_raw($remote_version->url) : '';
        $res->package = isset($remote_version->package) ? esc_url_raw($remote_version->package) : '';
        
        $res->tested = isset($remote_version->tested) ? sanitize_text_field($remote_version->tested) : '';
        $res->requires_php = isset($remote_version->requires_php) ? sanitize_text_field($remote_version->requires_php) : '';

        return $res;
    }

    /**
     * Map remote version data to WordPress plugin info object format.
     *
     * @param object $remote_version The remote version object.
     * @return stdClass The mapped info object.
     */
    protected function map_remote_info_data( $remote_version ) {
        $res = new stdClass();
        $res->name = isset($remote_version->name) ? sanitize_text_field($remote_version->name) : 'WordPress 中文補完計劃';
        $res->slug = $this->plugin_slug;
        $res->version = isset($remote_version->new_version) ? sanitize_text_field($remote_version->new_version) : '';
        $res->author = isset($remote_version->author) ? sanitize_text_field($remote_version->author) : '<a href="https://gordon168.com">Gordon168</a>';
        $res->author_profile = 'https://gordon168.com';
        $res->requires = isset($remote_version->requires) ? sanitize_text_field($remote_version->requires) : '5.0';
        $res->tested = isset($remote_version->tested) ? sanitize_text_field($remote_version->tested) : '';
        $res->requires_php = isset($remote_version->requires_php) ? sanitize_text_field($remote_version->requires_php) : '';
        $res->download_link = isset($remote_version->package) ? esc_url_raw($remote_version->package) : '';
        $res->trunk = isset($remote_version->package) ? esc_url_raw($remote_version->package) : '';
        $res->last_updated = isset($remote_version->last_updated) ? sanitize_text_field($remote_version->last_updated) : '';
        $res->sections = array(
            'description' => isset($remote_version->sections->description) ? wp_kses_post($remote_version->sections->description) : '',
            'installation' => isset($remote_version->sections->installation) ? wp_kses_post($remote_version->sections->installation) : '',
            'changelog' => isset($remote_version->sections->changelog) ? wp_kses_post($remote_version->sections->changelog) : '',
        );
        
        if ( isset($remote_version->banners) ) {
            $banners = (array) $remote_version->banners;
            $res->banners = array_map('esc_url_raw', $banners);
        }

        return $res;
    }

	/**
	 * Request update check from the API.
	 *
	 * @return object|bool The remote version object or false on failure.
	 */
	protected function request_update_check() {
        // Check cache first (cache for 12 hours)
        $cached = get_transient( $this->cache_key );
        if ( $cached !== false ) {
            return $cached;
        }

		$remote = wp_remote_post(
			$this->api_url,
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json',
                    // Don't send Authorization header here to keep it public-check friendly,
                    // but DO send api_key in body for package generation logic.
				),
                'body' => array(
                    'slug' => $this->plugin_slug,
                    'version' => $this->version,
                    'api_key' => $this->api_key,
                    'site_url' => get_site_url(),
                ),
			)
		);

		if ( 
            is_wp_error( $remote ) 
            || 200 !== wp_remote_retrieve_response_code( $remote ) 
            || empty( wp_remote_retrieve_body( $remote ) ) 
        ) {
			return false;
		}

		$remote = json_decode( wp_remote_retrieve_body( $remote ) );

        if ( json_last_error() !== JSON_ERROR_NONE || ! is_object( $remote ) ) {
            return false;
        }

        $remote = $this->sanitize_remote_version_payload( $remote );
        if ( false === $remote ) {
            return false;
        }

        // Cache the result
        set_transient( $this->cache_key, $remote, 12 * HOUR_IN_SECONDS );

		return $remote;
	}

    /**
     * Sanitizes and validates the remote version payload.
     *
     * @param object $remote Raw decoded JSON from the API.
     * @return object|false Sanitized payload or false if invalid.
     */
    protected function sanitize_remote_version_payload( $remote ) {
        if ( ! is_object( $remote ) ) {
            return false;
        }

        $new_version = isset( $remote->new_version ) ? $this->sanitize_version_string( $remote->new_version ) : '';
        if ( empty( $new_version ) ) {
            return false;
        }

        $safe            = new stdClass();
        $safe->new_version = $new_version;

        $safe->plugin = $this->sanitize_plugin_field( $remote->plugin ?? '' );
        if ( empty( $safe->plugin ) ) {
            $safe->plugin = $this->plugin_slug . '/' . $this->plugin_slug . '.php';
        }

        $safe->url = $this->sanitize_https_url( $remote->url ?? '' );
        $safe->package = $this->sanitize_package_url( $remote->package ?? '' );
        $safe->tested = isset( $remote->tested ) ? sanitize_text_field( $remote->tested ) : '';
        $safe->requires_php = isset( $remote->requires_php ) ? sanitize_text_field( $remote->requires_php ) : '';

        // Optional info fields for the plugin details popup.
        $safe->name         = isset( $remote->name ) ? sanitize_text_field( $remote->name ) : '';
        $safe->author       = isset( $remote->author ) ? sanitize_text_field( $remote->author ) : '';
        $safe->requires     = isset( $remote->requires ) ? sanitize_text_field( $remote->requires ) : '';
        $safe->last_updated = isset( $remote->last_updated ) ? sanitize_text_field( $remote->last_updated ) : '';

        if ( isset( $remote->sections ) && is_object( $remote->sections ) ) {
            $safe->sections = (object) array(
                'description'  => isset( $remote->sections->description ) ? wp_kses_post( $remote->sections->description ) : '',
                'installation' => isset( $remote->sections->installation ) ? wp_kses_post( $remote->sections->installation ) : '',
                'changelog'    => isset( $remote->sections->changelog ) ? wp_kses_post( $remote->sections->changelog ) : '',
            );
        }

        if ( isset( $remote->banners ) && is_object( $remote->banners ) ) {
            $banners = array();
            foreach ( (array) $remote->banners as $key => $value ) {
                $banners[ sanitize_key( $key ) ] = $this->sanitize_https_url( $value );
            }
            $safe->banners = (object) $banners;
        }

        return $safe;
    }

    /**
     * Ensures version string stays within a safe format for version_compare.
     *
     * @param mixed $version Raw version input.
     * @return string Sanitized version or empty string if invalid.
     */
    protected function sanitize_version_string( $version ) {
        if ( ! is_scalar( $version ) ) {
            return '';
        }

        $version = trim( (string) $version );
        // Allow semantic-like versions: digits, dots, dashes, plus.
        if ( ! preg_match( '/^[0-9A-Za-z\\.\\-\\+]+$/', $version ) ) {
            return '';
        }

        return $version;
    }

    /**
     * Sanitizes plugin field ensuring it references this plugin only.
     *
     * @param mixed $plugin Raw plugin field.
     * @return string Sanitized plugin path or empty string.
     */
    protected function sanitize_plugin_field( $plugin ) {
        if ( ! is_scalar( $plugin ) ) {
            return '';
        }

        $plugin = sanitize_text_field( (string) $plugin );

        // Only accept entries that point to this plugin slug.
        if ( strpos( $plugin, $this->plugin_slug ) !== 0 ) {
            return '';
        }

        return $plugin;
    }

    /**
     * Validates and sanitizes a general HTTPS URL.
     *
     * @param mixed $url Raw URL.
     * @return string Sanitized URL or empty string.
     */
    protected function sanitize_https_url( $url ) {
        if ( ! is_scalar( $url ) ) {
            return '';
        }

        $url = trim( (string) $url );
        if ( empty( $url ) ) {
            return '';
        }

        $validated = wp_http_validate_url( $url );
        if ( false === $validated ) {
            return '';
        }

        $parts = wp_parse_url( $validated );
        if ( empty( $parts['scheme'] ) || strtolower( $parts['scheme'] ) !== 'https' ) {
            return '';
        }

        return esc_url_raw( $validated );
    }

    /**
     * Validates and sanitizes package URL ensuring it matches the API host.
     *
     * @param mixed $url Raw URL.
     * @return string Sanitized URL or empty string.
     */
    protected function sanitize_package_url( $url ) {
        $url = $this->sanitize_https_url( $url );
        if ( empty( $url ) ) {
            return '';
        }

        // If we know the API host, ensure packages are served from the same host.
        if ( ! empty( $this->api_host ) ) {
            $parts = wp_parse_url( $url );
            if ( empty( $parts['host'] ) || strtolower( $parts['host'] ) !== strtolower( $this->api_host ) ) {
                return '';
            }
        }

        return $url;
    }
}
