<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class I10n_Push_Subscriber_Admin {

    private $plugin_name;
    private $version;
    private $api_key;
    private $updater; // New property to hold the Updater instance
    private $refresh_flag_option = 'i10n_push_available_items_refresh';

    public function __construct( $plugin_name, $version, $updater ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->api_key = get_option( 'i10n_push_api_key', '' );
        $this->updater = $updater; // Assign the Updater instance

        // Register AJAX actions
        add_action( 'wp_ajax_i10n_push_force_update_translations', array( $this, 'handle_force_update_translations_ajax' ) );
        add_action( 'wp_ajax_i10n_push_reapply_site', array( $this, 'handle_reapply_site_ajax' ) );
        add_action( 'wp_ajax_i10n_push_check_tos_consent', array( $this, 'handle_check_tos_consent_ajax' ) );
    }

    private function rate_limit_admin_action( $action, $max_per_user, $max_per_ip, $window_seconds ) {
        $user_id = get_current_user_id();
        $ip_raw = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $ip_hash = $ip_raw ? md5( $ip_raw ) : 'unknown';
        $window = $window_seconds > 0 ? (int) $window_seconds : 60;

        $user_key = 'i10n_push_rl_' . $action . '_user_' . $user_id;
        $ip_key = 'i10n_push_rl_' . $action . '_ip_' . $ip_hash;

        $user_count = (int) get_transient( $user_key );
        $ip_count = (int) get_transient( $ip_key );

        if ( $user_count >= $max_per_user || $ip_count >= $max_per_ip ) {
            return __( '請稍後再試，操作過於頻繁。', 'i10n-push-subscriber' );
        }

        $user_count++;
        $ip_count++;
        set_transient( $user_key, $user_count, $window );
        set_transient( $ip_key, $ip_count, $window );

        return false;
    }

    public function mark_items_refresh_needed() {
        delete_transient( 'i10n_push_available_items' );
        update_option( $this->refresh_flag_option, 1 );
    }

    public function maybe_refresh_available_items() {
        if ( ! get_option( $this->refresh_flag_option ) ) {
            return;
        }

        $connection_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        if ( 'connected' !== $connection_status ) {
            return;
        }

        $this->get_available_items();
        delete_option( $this->refresh_flag_option );
    }

    public function maybe_trigger_activation_update() {
        if ( ! get_option( 'i10n_push_force_update_on_activation' ) ) {
            return;
        }

        $connection_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        $api_key = get_option( 'i10n_push_api_key', '' );
        if ( 'connected' !== $connection_status || empty( $api_key ) ) {
            return;
        }

        $this->updater->check_for_updates( null, true );
        delete_option( 'i10n_push_force_update_on_activation' );
    }

    public function add_admin_menu() {
        add_options_page(
            __( '中文補完計劃', 'i10n-push-subscriber' ),
            __( '中文補完計劃', 'i10n-push-subscriber' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_settings_page' )
        );
    }

    public function display_settings_page() {
        $page_notices = array();

        // Check for and display API errors stored in a transient.
        $api_error = get_transient( 'i10n_push_api_error' );
        if ( $api_error && is_wp_error( $api_error ) ) {
            $page_notices[] = array(
                'type' => 'error',
                'html' => sprintf(
                    '<p><strong>%s</strong></p><p>%s</p>',
                    esc_html__( 'API 連線錯誤', 'i10n-push-subscriber' ),
                    sprintf(
                        /* translators: %s: API error message. */
                        esc_html__( '無法從伺服器取得可用項目清單。請檢查您的主機環境或聯絡我們。錯誤詳情：%s', 'i10n-push-subscriber' ),
                        '<br><pre>' . esc_html( $api_error->get_error_message() ) . '</pre>'
                    )
                ),
            );
            delete_transient( 'i10n_push_api_error' ); // Clear the transient after displaying
        }

        // Display success notice after refresh
        if ( isset( $_GET['message'] ) && 'refreshed' === $_GET['message'] ) {
            $page_notices[] = array(
                'type' => 'success',
                'html' => '<p>' . esc_html__( '清單已成功更新。', 'i10n-push-subscriber' ) . '</p>',
            );
        }

        $connection_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        $is_revoked        = ( 'revoked' === $connection_status );
        $is_reapply_pending = ( 'reapply_pending' === $connection_status );

        if ( $is_revoked ) {
            $page_notices[] = array(
                'type' => 'error',
                'html' => '<p><strong>' . esc_html__( '網站連接已被撤銷', 'i10n-push-subscriber' ) . '</strong></p><p>' . esc_html__( '您的網站因不符合使用條款而被撤銷連接，服務已停用。請聯繫我們或重新提交連接申請。', 'i10n-push-subscriber' ) . '</p>',
            );
        }
        if ( $is_reapply_pending ) {
            $page_notices[] = array(
                'type' => 'warning',
                'html' => '<p><strong>' . esc_html__( '已提交重新申請，等待審核', 'i10n-push-subscriber' ) . '</strong></p><p>' . esc_html__( '我們已收到您的重新申請，審核通過後會寄送驗證連結，請耐心等待。', 'i10n-push-subscriber' ) . '</p>',
            );
        }

        if ( 'connected' === $connection_status ) {
            // --- Logic for CONNECTED state ---

            // --- Temporary: Manual Confirmation Logic for Admins ---
            if ( isset( $_GET['confirm_subscriptions'] ) && 'true' === $_GET['confirm_subscriptions'] && current_user_can( 'manage_options' ) ) {
                $rate_limit_error = $this->rate_limit_admin_action( 'confirm_subscriptions', 3, 10, MINUTE_IN_SECONDS );
                if ( $rate_limit_error ) {
                    $page_notices[] = array(
                        'type' => 'error',
                        'html' => '<p>' . esc_html( $rate_limit_error ) . '</p>',
                    );
                } else {
                $confirm_nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
                if ( ! wp_verify_nonce( $confirm_nonce, 'i10n_push_confirm_subscriptions' ) ) {
                    $page_notices[] = array(
                        'type' => 'error',
                        'html' => '<p>' . esc_html__( '無效的請求，請重新整理後再試。', 'i10n-push-subscriber' ) . '</p>',
                    );
                } else {
                $pending_subscriptions = get_option( 'i10n_push_pending_subscriptions', array() );
                $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );

                if ( ! empty( $pending_subscriptions ) ) {
                    $expiration_date = date( 'Y-m-d', strtotime( '+1 year' ) );
                    foreach ( $pending_subscriptions as $domain ) {
                        $subscribed_items[ $domain ] = array(
                            'status'   => 'active',
                            'expires'  => $expiration_date,
                        );
                    }
                    update_option( 'i10n_push_subscribed_items', $subscribed_items );
                    
                    // Trigger an update check for the newly confirmed subscriptions
                    $this->updater->check_for_updates( $pending_subscriptions );

                    delete_option( 'i10n_push_pending_subscriptions' );
                    // Redirect to clean the URL
                    $redirect_url = admin_url( 'options-general.php?page=' . $this->plugin_name . '&message=confirmed' );
                    wp_safe_redirect( $redirect_url );
                    exit;
                }
                }
            }
            }
            
            // --- Force Update Check Logic ---
            if ( isset( $_GET['force_check'] ) && 'true' === $_GET['force_check'] && current_user_can( 'manage_options' ) ) {
                $rate_limit_error = $this->rate_limit_admin_action( 'force_check', 3, 10, MINUTE_IN_SECONDS );
                if ( $rate_limit_error ) {
                    $page_notices[] = array(
                        'type' => 'error',
                        'html' => '<p>' . esc_html( $rate_limit_error ) . '</p>',
                    );
                } else {
                $force_check_nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
                if ( ! wp_verify_nonce( $force_check_nonce, 'i10n_push_force_check' ) ) {
                    $page_notices[] = array(
                        'type' => 'error',
                        'html' => '<p>' . esc_html__( '無效的請求，請重新整理後再試。', 'i10n-push-subscriber' ) . '</p>',
                    );
                } else {
                // Check updates for all subscribed items
                $this->updater->check_for_updates();
                
                // Display success notice
                ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( '強制更新檢查已觸發。請檢查翻譯檔案是否已更新。', 'i10n-push-subscriber' ); ?></p>
                </div>
                <?php
                }
            }
            }

            // --- Data Fetching and Processing Logic ---
            $available_items_data = $this->get_available_items(); // This now returns ['plugins' => [...], 'themes' => [...]]
            $contact_support_url  = $available_items_data['contact_support_url'] ?? '';
            $managed_free_credit   = isset( $available_items_data['managed_free_credit'] ) ? (float) $available_items_data['managed_free_credit'] : 0;
            $designated_free_first_year_eligible = isset( $available_items_data['designated_free_first_year_eligible'] )
                ? (bool) $available_items_data['designated_free_first_year_eligible']
                : true;
            // If API call detected revoked status during this request, switch to revoked view.
            if ( 'revoked' === get_option( 'i10n_push_connection_status', 'disconnected' ) ) {
                ?>
                <div class="notice notice-error">
                    <p><strong><?php esc_html_e( '網站連接已被撤銷', 'i10n-push-subscriber' ); ?></strong></p>
                    <p><?php esc_html_e( '您的網站因不符合使用條款而被撤銷連接，服務已停用。請聯繫我們或重新提交連接申請。', 'i10n-push-subscriber' ); ?></p>
                </div>
                <?php
                $admin_email = get_option( 'admin_email' );
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/i10n-push-subscriber-admin-connect.php';
                return;
            }
            $plugins = $available_items_data['plugins'] ?? [];
            $themes = $available_items_data['themes'] ?? [];

            // Exclude this plugin itself from the list
            $plugins = array_filter($plugins, function($plugin) {
                return $plugin['text_domain'] !== 'i10n-push-subscriber';
            });

            // Sort plugins by name A-Z
            usort($plugins, function($a, $b) {
                return strcmp($a['name'], $b['name']);
            });

            // Sort themes by name A-Z
            usort($themes, function($a, $b) {
                return strcmp($a['name'], $b['name']);
            });

            // Flag child themes for special display handling.
            $child_theme_map = array();
            $installed_themes_map = wp_get_themes();
            foreach ( $installed_themes_map as $theme_slug => $theme_data ) {
                $parent_theme = $theme_data->parent();
                if ( $parent_theme ) {
                    $child_theme_map[ $theme_slug ] = $parent_theme->get( 'Name' );
                }
            }
            foreach ( $themes as &$theme ) {
                if ( isset( $theme['text_domain'] ) && isset( $child_theme_map[ $theme['text_domain'] ] ) ) {
                    $theme['is_child_theme'] = true;
                    $theme['child_theme_parent'] = $child_theme_map[ $theme['text_domain'] ];
                }
            }
            unset($theme);

            // --- Inject PO/POT File URLs ---
            // Extract text domains for scanning
            $plugin_domains = array_column($plugins, 'text_domain');
            $theme_domains = array_column($themes, 'text_domain');
            $all_domains = array_merge($plugin_domains, $theme_domains);
            
            // Scan for translation files
            $file_urls = $this->scan_translation_files($all_domains);

            // Inject file URLs into plugin items
            foreach ($plugins as &$plugin) {
                if (isset($file_urls[$plugin['text_domain']])) {
                    $plugin['po_file_url'] = $file_urls[$plugin['text_domain']];
                }
            }
            unset($plugin); // Break reference

            // Inject file URLs into theme items
            foreach ($themes as &$theme) {
                if (isset($file_urls[$theme['text_domain']])) {
                    $theme['po_file_url'] = $file_urls[$theme['text_domain']];
                }
            }
            unset($theme); // Break reference

            // --- Calculate Current Subscription Summary ---
            $all_items = array_merge($plugins, $themes);
            $subscribed_items_count = 0;
            $total_original_cost = 0;
            $total_discount = 0;
            $designated_free_used = 0;
            $designated_free_limit = $designated_free_first_year_eligible ? 2 : 0;

            $designated_free_items_prices = []; // Collect prices of subscribed designated-free items

            foreach ($all_items as $item) {
                if ($item['is_subscribed']) {
                    $subscribed_items_count++;
                    $price = (int) ($item['price'] ?? 0);
                    $total_original_cost += $price; // Sum up all original subscribed prices

                    if ($item['source'] === 'designated-free' && $price > 0) {
                        $designated_free_items_prices[] = $price;
                    }
                }
            }

            // Sort designated-free items by price descending to apply discount to most expensive ones first
            rsort($designated_free_items_prices);

            // Apply discount to the top N designated-free items
            foreach ($designated_free_items_prices as $price) {
                if ($designated_free_used < $designated_free_limit) {
                    $total_discount += $price;
                    $designated_free_used++;
                } else {
                    // Once limit is reached, no more discounts for designated-free items
                    break;
                }
            }

            // Renewal price should include designated-free items after first year.
            $total_annual_cost = $total_original_cost;

            // The view file will now have access to the data
            $page_notices = $page_notices;
            $managed_free_credit = $managed_free_credit;
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/i10n-push-subscriber-admin-display.php';

        } else {
            // --- Logic for DISCONNECTED or PENDING state ---
            $admin_email = get_option( 'admin_email' );
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/i10n-push-subscriber-admin-connect.php';
        }
    }

    private function calculate_subscription_summary( $subscribed_items, $available_items_lookup, $designated_free_first_year_eligible ) {
        $summary = array(
            'total_subscribed_count' => 0,
            'total_original_cost'    => 0,
            'total_discount'         => 0,
            'total_annual_cost'      => 0,
        );

        if ( is_array( $subscribed_items ) && ! empty( $subscribed_items ) ) {
            $summary['total_subscribed_count'] = count( $subscribed_items );
            $designated_free_used = 0;
            foreach ( $subscribed_items as $domain => $details ) {
                if ( isset( $available_items_lookup[ $domain ] ) ) {
                    $item = $available_items_lookup[ $domain ];
                    $summary['total_original_cost'] += (int) $item['price'];
                    if ( $designated_free_first_year_eligible && 'designated-free' === $item['source'] && $designated_free_used < 2 ) {
                        $designated_free_used++;
                        $summary['total_discount'] += (int) $item['price'];
                    }
                }
            }
            // Renewal price should include designated-free items after first year.
            $summary['total_annual_cost'] = $summary['total_original_cost'];
        }

        return $summary;
    }

    public function get_available_items() {
        $transient_key = 'i10n_push_available_items';

        // Guard: never call server API until the site is actually connected and has an API key.
        // This prevents false "pending verification" states on fresh installs.
        $current_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        if ( empty( $this->api_key ) ) {
            if ( 'connected' === $current_status ) {
                update_option( 'i10n_push_connection_status', 'disconnected' );
            }
            return array( 'plugins' => [], 'themes' => [] );
        }
        if ( 'connected' !== $current_status ) {
            return array( 'plugins' => [], 'themes' => [] );
        }

        // Always scan installed items, as this is needed for the API request.
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $installed_plugins = get_plugins();
        $installed_themes = wp_get_themes();

        $installed_items_for_api = array();

        if ( ! function_exists( 'is_plugin_active' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        // Process plugins
        foreach ( $installed_plugins as $plugin_path => $plugin_data ) {
            $text_domain = dirname( $plugin_path );
            if ( '.' === $text_domain ) {
                $text_domain = preg_replace( '/\.(php)$/', '', basename( $plugin_path ) );
            }
            $is_active = is_plugin_active( $plugin_path );
            $installed_items_for_api[] = array(
                'text_domain' => $text_domain,
                'name'        => $plugin_data['Name'],
                'type'        => 'plugin',
                'version'     => $plugin_data['Version'],
                'is_active'   => (bool) $is_active,
            );
        }

        // Process themes
        $active_theme = wp_get_theme();
        $active_stylesheet = $active_theme->get_stylesheet();
        $active_template = $active_theme->get_template();
        $active_parent = $active_theme->parent();
        $active_parent_slug = $active_parent ? $active_parent->get_stylesheet() : '';
        foreach ( $installed_themes as $theme_slug => $theme_data ) {
            $is_active = (
                $theme_slug === $active_stylesheet
                || $theme_slug === $active_template
                || ( $active_parent_slug && $theme_slug === $active_parent_slug )
            );
            $installed_items_for_api[] = array(
                'text_domain' => $theme_slug,
                'name'        => $theme_data->get( 'Name' ),
                'type'        => 'theme',
                'version'     => $theme_data->get( 'Version' ),
                'is_active'   => (bool) $is_active,
            );
        }

        // Production logic: Fetch from API
        $cached_items = get_transient( $transient_key );
        $last_status_check = (int) get_option( 'i10n_push_last_status_check', 0 );
        $status_check_ttl = 5 * MINUTE_IN_SECONDS;
        // If cache exists but缺少 managed_free_credit，強制刷新以取得最新資料
        if ( false !== $cached_items && isset( $cached_items['managed_free_credit'] ) ) {
            if ( ( time() - $last_status_check ) < $status_check_ttl ) {
                return $cached_items;
            }
        }
        // Stale cache: clear and refetch
        if ( false !== $cached_items ) {
            delete_transient( $transient_key );
        }

        $api_url = I10N_PUSH_API_URL . '/get-available-items';
        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 30,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type'  => 'application/json',
                ),
                'body'    => wp_json_encode( array( 'site_url' => get_site_url(), 'installed_items' => $installed_items_for_api ) ),
            )
        );

        if ( is_wp_error( $response ) ) {
            // Save the error object to a transient to display it in the admin notice
            set_transient( 'i10n_push_api_error', $response, 60 );
            set_transient( $transient_key, array('plugins' => [], 'themes' => []), 5 * MINUTE_IN_SECONDS ); // Cache failure for 5 mins
            update_option( 'i10n_push_last_status_check', time() );
            return array('plugins' => [], 'themes' => []);
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );

        // --- Start Debug Logging ---
        // Always log the raw response from the server for debugging purposes.
        i10n_push_log( "[i10n-push-subscriber] DEBUG: API Response Code: " . $response_code );
        i10n_push_log( "[i10n-push-subscriber] DEBUG: API Response Body: " . $response_body );
        // --- End Debug Logging ---

        if ( 200 !== $response_code ) {
            // If unauthorized/forbidden while connected, treat as revoked connection to reflect in UI.
            if ( in_array( $response_code, array( 401, 403 ), true ) ) {
                update_option( 'i10n_push_connection_status', 'revoked' );
                delete_option( 'i10n_push_api_key' );
            }
            i10n_push_log( "[i10n-push-subscriber] API error on get_available_items. Status: {$response_code}. Body: " . $response_body );
            set_transient( $transient_key, array('plugins' => [], 'themes' => []), 5 * MINUTE_IN_SECONDS ); // Cache failure for 5 mins
            update_option( 'i10n_push_last_status_check', time() );
            return array('plugins' => [], 'themes' => []);
        }

        $items = json_decode( $response_body, true );

        if ( is_null( $items ) || !isset($items['plugins']) || !isset($items['themes']) ) {
            i10n_push_log( "[i10n-push-subscriber] Failed to decode JSON or invalid structure from get_available_items. Body: " . $response_body );
            set_transient( $transient_key, array('plugins' => [], 'themes' => []), 5 * MINUTE_IN_SECONDS ); // Cache failure for 5 mins
            update_option( 'i10n_push_last_status_check', time() );
            return array('plugins' => [], 'themes' => []);
        }

        // Normalize managed free credit if provided
        $items['managed_free_credit'] = isset( $items['managed_free_credit'] ) ? (float) $items['managed_free_credit'] : 0;

        // If server explicitly flags revoked, reflect locally
        if ( isset( $items['status'] ) && 'revoked' === $items['status'] ) {
            update_option( 'i10n_push_connection_status', 'revoked' );
            delete_option( 'i10n_push_api_key' );
            set_transient( $transient_key, array('plugins' => [], 'themes' => []), 5 * MINUTE_IN_SECONDS );
            update_option( 'i10n_push_last_status_check', time() );
            return array('plugins' => [], 'themes' => []);
        }
        
        // --- Sanitize API Response ---
        $items['plugins'] = array_map( array( $this, 'sanitize_item_data' ), $items['plugins'] );
        $items['themes']  = array_map( array( $this, 'sanitize_item_data' ), $items['themes'] );
        // -----------------------------
        
        // --- Sync Local Subscription State ---
        // Automatically update local subscription status based on server response
        $current_subscribed_items = get_option( 'i10n_push_subscribed_items', array() ); // Capture existing subscriptions BEFORE update
        $new_subscribed_items = array();
        $newly_subscribed_domains = [];

        $all_received_items = array_merge( $items['plugins'], $items['themes'] );

        foreach ( $all_received_items as $item ) {
            if ( isset( $item['is_subscribed'] ) && $item['is_subscribed'] === true ) {
                $text_domain = $item['text_domain'];
                
                // Preserve existing expiration date if possible, or use server provided one if available (future improvement)
                // For now, we just mark it as active.
                $expires = isset( $current_subscribed_items[$text_domain]['expires'] ) ? $current_subscribed_items[$text_domain]['expires'] : date('Y-m-d', strtotime('+1 year')); 
                if ( isset( $item['expires_at'] ) && !empty( $item['expires_at'] ) ) {
                     $expires = $item['expires_at'];
                }

                $new_subscribed_items[$text_domain] = array(
                    'status'  => 'active',
                    'expires' => $expires,
                );

                // Identify newly subscribed items (not in current_subscribed_items, but now in new_subscribed_items)
                if ( ! isset( $current_subscribed_items[$text_domain] ) ) {
                    $newly_subscribed_domains[] = $text_domain;
                }
            }
        }
        
        // Check if there are changes to avoid unnecessary DB writes (simple count check + key diff)
        if ( count( $current_subscribed_items ) !== count( $new_subscribed_items ) || array_diff_key( $current_subscribed_items, $new_subscribed_items ) !== array_diff_key( $new_subscribed_items, $current_subscribed_items ) ) {
             update_option( 'i10n_push_subscribed_items', $new_subscribed_items );

        }
        
        $free_items = array();
        foreach ( $all_received_items as $received_item ) {
            if ( isset( $received_item['tier'] ) && 'free' === $received_item['tier'] ) {
                $free_items[] = $received_item['text_domain'];
            }
        }
        update_option( 'i10n_push_free_items', $free_items );

        // If there are newly subscribed items, trigger an immediate update check for them.
        if ( ! empty( $newly_subscribed_domains ) ) {
            i10n_push_log( '[i10n-push-subscriber] Detected newly subscribed items. Triggering immediate translation update.' );
            $this->updater->check_for_updates( $newly_subscribed_domains, true );
        }
        // -------------------------------------
        
        set_transient( $transient_key, $items, 12 * HOUR_IN_SECONDS );
        update_option( 'i10n_push_last_status_check', time() );

        // Preserve support URL if provided
        if ( isset( $items['contact_support_url'] ) ) {
            $items['contact_support_url'] = esc_url_raw( $items['contact_support_url'] );
        }
        // Preserve managed free credit
        if ( isset( $items['managed_free_credit'] ) ) {
            $items['managed_free_credit'] = (float) $items['managed_free_credit'];
        }

        return $items;
    }

    public function enqueue_scripts( $hook ) {
        // Always enqueue the main admin script, as the connect notice can appear on any page.
        wp_enqueue_script( $this->plugin_name . '-admin', plugin_dir_url( dirname( __FILE__ ) ) . 'admin/js/i10n-push-subscriber-admin.js', array( 'jquery' ), time(), true );

        $connection_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        $available_items_data = array();
        $managed_free_credit = 0;

        // Only call /get-available-items when we are on our settings page AND the site is already connected.
        if ( 'settings_page_' . $this->plugin_name === $hook && 'connected' === $connection_status ) {
            $available_items_data = $this->get_available_items();
            $managed_free_credit = isset( $available_items_data['managed_free_credit'] ) ? (float) $available_items_data['managed_free_credit'] : 0;
        }

        // Prepare the basic data needed on all pages for AJAX calls.
        $script_data = array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'i10n_push_nonce' ),
            'site_url' => get_site_url(), // Add site_url for the polling request
            'plugin_name' => $this->plugin_name, // Add plugin_name for redirects
            'managed_free_credit' => $managed_free_credit,
        );

        // If we are on the plugin's settings page, add the extra data.
        if ( 'settings_page_' . $this->plugin_name === $hook ) {
            $css_path = plugin_dir_path( dirname( __FILE__ ) ) . 'admin/css/i10n-push-subscriber-admin.css';
            $css_version = file_exists( $css_path ) ? (string) filemtime( $css_path ) : $this->version;
            wp_enqueue_style( $this->plugin_name, plugin_dir_url( dirname( __FILE__ ) ) . 'admin/css/i10n-push-subscriber-admin.css', array(), $css_version, 'all' );
            
            $script_data['available_items']  = $available_items_data;
            $script_data['subscribed_items'] = get_option( 'i10n_push_subscribed_items', array() );

            $active_items = array();
            $all_items = array_merge(
                $available_items_data['plugins'] ?? array(),
                $available_items_data['themes'] ?? array()
            );
            foreach ( $all_items as $item ) {
                if ( ! empty( $item['is_active'] ) && ! empty( $item['text_domain'] ) ) {
                    $active_items[] = $item['text_domain'];
                }
            }
            $script_data['active_items'] = $active_items;
        }

        // Localize the script with the combined data.
        wp_localize_script( $this->plugin_name . '-admin', 'i10n_push_data', $script_data );
    }

    /**
     * Sanitizes the data for a single plugin or theme item received from the API.
     *
     * @param array $item The item data to sanitize.
     * @return array The sanitized item data.
     */
    private function sanitize_item_data( $item ) {
        if ( ! is_array( $item ) ) {
            return [];
        }

        foreach ( $item as $key => $value ) {
            if ( is_string( $value ) ) {
                $item[ $key ] = sanitize_text_field( $value );
            }
        }

        return $item;
    }

    public function handle_check_verification_ajax() {
        check_ajax_referer( 'i10n_push_nonce', '_wpnonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        $api_url = I10N_PUSH_API_URL . '/check-verification-status';
        $body    = array(
            'site_url' => get_site_url(),
        );

        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 30,
                'headers' => array( 'Content-Type' => 'application/json' ),
                'body'    => wp_json_encode( $body ),
            )
        );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => 'Polling request failed: ' . $response->get_error_message() ) );
            return;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body_raw = wp_remote_retrieve_body( $response );
        $response_body = json_decode( $response_body_raw, true );

        if ( is_null( $response_body ) ) {
            wp_send_json_error( array( 'message' => '無法解析伺服器回應的 JSON。', 'response' => $response_body_raw ) );
            return;
        }

        // 標準狀態處理
        $status = $response_body['status'] ?? '';
        if ( 202 === $response_code ) {
            update_option( 'i10n_push_connection_status', 'pending' );
            if ( ! get_option( 'i10n_push_pending_verification_at' ) ) {
                update_option( 'i10n_push_pending_verification_at', time() );
            }
            wp_send_json_success( array(
                'status'  => 'pending',
                'message' => $response_body['message'] ?? '驗證郵件已寄出，請至信箱點擊連結完成啟用。',
            ) );
            return;
        }

        if ( 200 === $response_code && 'verified' === $status ) {
            update_option( 'i10n_push_connection_status', 'connected' );
            delete_option( 'i10n_push_pending_verification_at' );
            wp_send_json_success( array(
                'status'  => 'connected',
                'message' => $response_body['message'] ?? '網站已驗證完成並取得 API 金鑰。',
                'api_key' => $response_body['api_key'] ?? '',
            ) );
            return;
        }

        if ( 200 === $response_code && isset( $response_body['api_key'] ) ) {
            update_option( 'i10n_push_api_key', sanitize_text_field( $response_body['api_key'] ) );
            update_option( 'i10n_push_connection_status', 'connected' );
            delete_option( 'i10n_push_pending_verification_at' );
            wp_send_json_success( array(
                'status'  => 'connected',
                'message' => $response_body['message'] ?? '網站已連接完成。',
            ) );
            return;
        }

        $error_message = $response_body['message'] ?? "伺服器回應碼 {$response_code}";
        wp_send_json_error( array( 'message' => $error_message ), $response_code );
    }

    public function handle_connect_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        $tos_nonce = get_option( 'i10n_push_tos_nonce' );
        if ( empty( $tos_nonce ) ) {
            wp_send_json_error( array( 'message' => __( '請先閱讀並同意服務條款後再連接。', 'i10n-push-subscriber' ) ), 400 );
        }

        // Verify consent with API before initiating register-site.
        $consent_status_url = trailingslashit( I10N_PUSH_API_URL ) . 'tos/consent-status';
        $consent_response   = wp_remote_get(
            add_query_arg(
                array(
                    'site_url'   => get_site_url(),
                    'tos_nonce'  => $tos_nonce,
                ),
                $consent_status_url
            ),
            array( 'timeout' => 20 )
        );
        if ( is_wp_error( $consent_response ) ) {
            wp_send_json_error( array( 'message' => __( '無法檢查服務條款同意狀態，請稍後再試。', 'i10n-push-subscriber' ) ), 500 );
        }
        $consent_body = json_decode( wp_remote_retrieve_body( $consent_response ), true );
        if ( empty( $consent_body['consented'] ) ) {
            wp_send_json_error( array( 'message' => __( '請先閱讀並同意服務條款後再連接。', 'i10n-push-subscriber' ) ), 400 );
        }

        $api_url = I10N_PUSH_API_URL . '/register-site';
        $body    = array(
            'admin_email' => get_option( 'admin_email' ),
            'admin_display_name' => wp_get_current_user()->display_name,
            'site_name'   => get_option( 'blogname' ),
            'site_url'    => get_site_url(),
            'tos_nonce'   => $tos_nonce,
        );

        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 30,
                'headers' => array( 'Content-Type' => 'application/json' ),
                'body'    => wp_json_encode( $body ),
            )
        );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => __( '無法連接到伺服器：', 'i10n-push-subscriber' ) . $response->get_error_message() ) );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = json_decode( wp_remote_retrieve_body( $response ), true );

        // The server returns 202 Accepted to indicate the verification email has been sent.
        if ( 202 === $response_code ) {
            // The process has started. The frontend will now poll for completion.
            update_option( 'i10n_push_connection_status', 'pending' );
            update_option( 'i10n_push_pending_verification_at', time() );
            wp_send_json_success( array(
                'status' => 'pending',
                'message' => $response_body['message'] ?? __( '驗證郵件已寄出，請至您的信箱收信並點擊連結以完成啟用。', 'i10n-push-subscriber' )
            ) );
        } elseif ( 200 === $response_code && isset( $response_body['api_key'] ) ) {
            // The site is already connected or auto-verified, save the key immediately.
            update_option( 'i10n_push_api_key', sanitize_text_field( $response_body['api_key'] ) );
            update_option( 'i10n_push_connection_status', 'connected' );
            delete_option( 'i10n_push_pending_verification_at' );
            wp_send_json_success( array(
                'status' => 'connected',
                'message' => __( '網站已成功連接！', 'i10n-push-subscriber' )
            ) );
        } else {
            // Handle all other codes as errors.
            $error_message = isset( $response_body['message'] ) ? $response_body['message'] : __( '發生未知錯誤。', 'i10n-push-subscriber' );
            /* translators: 1: HTTP status code, 2: error message returned by server. */
            wp_send_json_error( array( 'message' => sprintf( __( '伺服器錯誤 (代碼: %1$d): %2$s', 'i10n-push-subscriber' ), $response_code, $error_message ) ) );
        }
    }

    public function handle_save_api_key_ajax() {
        check_ajax_referer( 'i10n_push_nonce', '_wpnonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        if ( isset( $_POST['api_key'] ) && ! empty( $_POST['api_key'] ) ) {
            $api_key = sanitize_text_field( $_POST['api_key'] );
            update_option( 'i10n_push_api_key', $api_key );
            update_option( 'i10n_push_connection_status', 'connected' );
            delete_option( 'i10n_push_pending_verification_at' );
            wp_send_json_success( array( 'message' => __( 'API 金鑰已儲存。', 'i10n-push-subscriber' ) ) );
        } else {
            wp_send_json_error( array( 'message' => __( '未提供 API 金鑰。', 'i10n-push-subscriber' ) ), 400 );
        }
    }

    public function handle_create_order_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );
        
        $subscriptions = isset( $_POST['subscriptions'] ) ? wp_unslash( (array) $_POST['subscriptions'] ) : array();
        $sanitized_subscriptions = array_map( 'sanitize_text_field', $subscriptions );
        $active_items = isset( $_POST['active_items'] ) ? wp_unslash( (array) $_POST['active_items'] ) : array();
        $sanitized_active_items = array_map( 'sanitize_text_field', $active_items );

        // Directly use the translation_files array sent from the client-side JavaScript.
        $translation_files = isset( $_POST['translation_files'] ) ? wp_unslash( (array) $_POST['translation_files'] ) : array();

        // In dev mode, we send a test email. The actual order is not created on the server.
        if ( defined( 'I10N_PUSH_DEV_MODE' ) && I10N_PUSH_DEV_MODE ) {
            $this->send_test_email( '待處理的訂單請求', '使用者提交了新的訂單請求', array( 
                '訂購項目' => $sanitized_subscriptions,
                '翻譯檔案連結' => $translation_files 
            ) );
            wp_send_json_success( array( 'message' => '測試模式：訂單請求已提交，等待審核中。' ) );
            return;
        }

        // --- Production Mode: Call /create-order endpoint ---
        $api_url = I10N_PUSH_API_URL . '/create-order';
        $body    = array(
            'site_url'      => get_site_url(),
            'subscriptions' => $sanitized_subscriptions,
            'translation_files' => $translation_files, // Pass the collected file URLs
            'active_items' => $sanitized_active_items,
        );

        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 45,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type'  => 'application/json',
                ),
                'body'    => wp_json_encode( $body ),
            )
        );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => __( '無法連接到訂單伺服器：', 'i10n-push-subscriber' ) . $response->get_error_message() ) );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( 200 !== $response_code ) {
            $error_message = isset( $response_body['message'] ) ? $response_body['message'] : __( '發生未知錯誤。', 'i10n-push-subscriber' );
            /* translators: 1: HTTP status code, 2: error message returned by server. */
            wp_send_json_error( array( 'message' => sprintf( __( '伺服器錯誤 (代碼: %1$d): %2$s', 'i10n-push-subscriber' ), $response_code, $error_message ) ) );
        }

        // On success, clear cached available items so reload shows pending_payment status.
        delete_transient( 'i10n_push_available_items' );
        wp_send_json_success( array( 'message' => __( '訂單已建立，請至您的信箱查收付款說明。', 'i10n-push-subscriber' ) ) );
    }

    /**
     * Scans plugin/theme directories for .po or .pot files.
     *
     * @param array $text_domains Array of text domains to scan.
     * @return array Array of file URLs keyed by text domain.
     */
    private function scan_translation_files( $text_domains ) {
        $file_urls = array();
        $site_url = get_site_url();

        // Get all plugins and themes to find paths
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        $themes = wp_get_themes();

        foreach ( $text_domains as $domain ) {
            $path = '';
            $url_base = '';
            $type = '';

            // Check plugins
            foreach ( $plugins as $plugin_path => $data ) {
                $plugin_domain = dirname( $plugin_path );
                if ( '.' === $plugin_domain ) {
                    $plugin_domain = preg_replace( '/\.(php)$/', '', basename( $plugin_path ) );
                }
                if ( $plugin_domain === $domain ) {
                    $path = WP_PLUGIN_DIR . '/' . dirname( $plugin_path );
                    $url_base = plugins_url( dirname( $plugin_path ) ); // Correctly gets URL base for plugin
                    $type = 'plugin';
                    break;
                }
            }

            // Check themes if not found in plugins
            if ( empty( $path ) ) {
                foreach ( $themes as $theme_slug => $data ) {
                    if ( $theme_slug === $domain ) {
                        $path = get_theme_root() . '/' . $theme_slug;
                        $url_base = get_theme_root_uri() . '/' . $theme_slug;
                        $type = 'theme';
                        break;
                    }
                }
            }

            if ( ! empty( $path ) ) {
                // Search order updated: 
                // 1. languages/{domain}-zh_TW.po
                // 2. languages/{domain}.po
                // 3. languages/{domain}.pot
                // 4. {domain}-zh_TW.po (root)
                // 5. {domain}.po (root)
                // 6. {domain}.pot (root)
                
                $candidates = array(
                    '/languages/' . $domain . '-zh_TW.po',
                    '/languages/' . $domain . '.po',
                    '/languages/' . $domain . '.pot',
                    '/' . $domain . '-zh_TW.po',
                    '/' . $domain . '.po',
                    '/' . $domain . '.pot',
                );

                $found_url = '';

                foreach ( $candidates as $candidate ) {
                    if ( file_exists( $path . $candidate ) ) {
                        $found_url = $url_base . $candidate;
                        break;
                    }
                }
                
                // If still not found, look for ANY .po file in languages/ first
                if ( empty( $found_url ) && is_dir( $path . '/languages' ) ) {
                    $po_files = glob( $path . '/languages/*.po' );
                    if ( ! empty( $po_files ) ) {
                        $found_url = $url_base . '/languages/' . basename( $po_files[0] );
                    } else {
                         // Then look for ANY .pot file
                        $pot_files = glob( $path . '/languages/*.pot' );
                        if ( ! empty( $pot_files ) ) {
                            $found_url = $url_base . '/languages/' . basename( $pot_files[0] );
                        }
                    }
                }

                if ( ! empty( $found_url ) ) {
                    $file_urls[ $domain ] = $found_url;
                }
            }
        }

        return $file_urls;
    }

    public function handle_request_quote_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );
        $text_domains = isset( $_POST['text_domains'] ) ? wp_unslash( $_POST['text_domains'] ) : array();
        $active_items = isset( $_POST['active_items'] ) ? wp_unslash( (array) $_POST['active_items'] ) : array();

        if ( empty( $text_domains ) ) {
            wp_send_json_error( array( 'message' => '無效的項目。' ) );
        }

        $sanitized_domains = array_map( 'sanitize_text_field', $text_domains );
        $sanitized_active_items = array_map( 'sanitize_text_field', $active_items );
        
        // Scan for translation files
        $translation_files = $this->scan_translation_files( $sanitized_domains );

        if ( defined( 'I10N_PUSH_DEV_MODE' ) && I10N_PUSH_DEV_MODE ) {
            $this->send_test_email( '批次請求報價', '使用者請求新項目報價', array( 
                '項目 Text Domains' => $sanitized_domains,
                '翻譯檔案連結' => $translation_files 
            ) );
            wp_send_json_success( array( 'message' => '測試模式：報價請求已記錄 (未發送到伺服器)。' ) );
            return;
        }

        $api_url = I10N_PUSH_API_URL . '/request-quote';
        $body = array( 
            'site_url' => get_site_url(), 
            'text_domains' => $sanitized_domains,
            'translation_files' => $translation_files, // Add file URLs to request
            'active_items' => $sanitized_active_items
        );
        
        $response = wp_remote_post( $api_url, array( 'body' => wp_json_encode( $body ), 'headers' => array( 'Authorization' => 'Bearer ' . $this->api_key, 'Content-Type' => 'application/json' ) ) );
        
        if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
            wp_send_json_error( array( 'message' => '請求報價失敗。' ) );
        }
        
        delete_transient( 'i10n_push_available_items' );
        wp_send_json_success( array( 'message' => '報價請求已送出。' ) );
    }

    public function handle_cancel_all_subscriptions_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        // Clear the API key and subscribed items locally
        delete_option( 'i10n_push_api_key' );
        delete_option( 'i10n_push_connection_status' );
        delete_option( 'i10n_push_pending_verification_at' );
        delete_option( 'i10n_push_tos_nonce' );
        delete_option( 'i10n_push_subscribed_items' );
        delete_option( 'i10n_push_pending_subscriptions' );
        delete_option( 'i10n_push_free_items' );
        delete_transient( 'i10n_push_available_items' ); // Clear cached items

        wp_send_json_success( array( 'message' => __( '所有訂閱已取消。頁面將會重新整理。', 'i10n-push-subscriber' ) ) );
    }

    /**
     * Handle AJAX request to get available translation items.
     *
     * @since    1.0.0
     */
    public function handle_get_available_items_ajax() {
        check_ajax_referer( 'i10n_push_get_available_items_nonce', '_wpnonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        $available_items = $this->get_available_items();
        $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );

        wp_send_json_success( array(
            'items' => $available_items,
            'subscribed_items' => $subscribed_items,
        ) );
    }

    public function handle_force_refresh_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        delete_transient( 'i10n_push_available_items' );

        wp_send_json_success( array( 'message' => __( '快取已清除。', 'i10n-push-subscriber' ) ) );
    }

    private function send_test_email( $subject, $title, $data ) {
        $to = 'i10n@gordon168.com';
        $subject = '[i10n Push Test] ' . $subject . ' - ' . get_site_url();
        $body = "<h1>{$title}</h1>";
        $body .= '<p><strong>網站網址:</strong> ' . get_site_url() . '</p>';
        $body .= '<p><strong>管理員 Email:</strong> ' . get_option( 'admin_email' ) . '</p>';
        
        foreach ( $data as $key => $value ) {
            $body .= '<p><strong>' . esc_html( $key ) . ':</strong></p>';
            if ( is_array( $value ) && ! empty( $value ) ) {
                $body .= '<ul>';
                foreach ( $value as $item ) {
                    $body .= '<li>' . esc_html( $item ) . '</li>';
                }
                $body .= '</ul>';
            } elseif ( ! is_array( $value ) ) {
                $body .= '<p>' . esc_html( $value ) . '</p>';
            } else {
                $body .= '<p>N/A</p>';
            }
        }

        $headers = array( 'Content-Type: text/html; charset=UTF-8' );
        wp_mail( $to, $subject, $body, $headers );
    }

    public function display_smart_notification() {
        $notice_transient = get_transient( 'i10n_push_smart_notice' );
        if ( empty( $notice_transient ) ) {
            return;
        }

        delete_transient( 'i10n_push_smart_notice' ); // Delete after displaying once

        $item_name = $notice_transient['name'];
        $item_info = $notice_transient['info'];
        $settings_page_url = admin_url( 'options-general.php?page=' . $this->plugin_name );

        ?>
        <div class="notice notice-info is-dismissible">
            <p>
                <strong><?php esc_html_e( 'i10n Push 提示', 'i10n-push-subscriber' ); ?></strong>:
                <?php /* translators: %s: item name detected on the site. */ ?>
                <?php printf( esc_html__( '偵測到您已安裝「%s」。', 'i10n-push-subscriber' ), '<strong>' . esc_html( $item_name ) . '</strong>' ); ?>
            </p>
            <p>
                <?php if ( isset( $item_info['price'] ) && $item_info['price'] == 0 ) : ?>
                    <?php esc_html_e( '此項目已包含在我們的指定免費清單中，專業翻譯將自動啟用。', 'i10n-push-subscriber' ); ?>
                    <a href="<?php echo esc_url( $settings_page_url ); ?>" class="button button-secondary">
                        <?php esc_html_e( '查看所有免費項目', 'i10n-push-subscriber' ); ?>
                    </a>
                <?php elseif ( isset( $item_info['source'] ) && 'commercial' === $item_info['source'] ) : ?>
                    <?php esc_html_e( '此為商業/專業版項目，訂閱專業翻譯的價格為 NT$200 / 年。', 'i10n-push-subscriber' ); ?>
                    <a href="<?php echo esc_url( $settings_page_url ); ?>" class="button button-primary">
                        <?php esc_html_e( '前往新增訂閱', 'i10n-push-subscriber' ); ?>
                    </a>
                <?php else : ?>
                    <?php /* translators: %d: yearly price in TWD. */ ?>
                    <?php printf( esc_html__( '訂閱此項目的專業翻譯價格為 NT$%d / 年。', 'i10n-push-subscriber' ), (int) $item_info['price'] ); ?>
                     <a href="<?php echo esc_url( $settings_page_url ); ?>" class="button button-primary">
                        <?php esc_html_e( '前往新增訂閱', 'i10n-push-subscriber' ); ?>
                    </a>
                <?php endif; ?>
            </p>
        </div>
        <?php
    }

    public function display_connect_notice() {
        // Do not show the notice if the connection is pending, already connected, or on our settings page.
        if ( get_option( 'i10n_push_connection_status' ) === 'pending' || ! empty( $this->api_key ) || ( isset( $_GET['page'] ) && $_GET['page'] === $this->plugin_name ) ) {
            return;
        }
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        $current_site = get_site_url();
        $tos_nonce    = get_option( 'i10n_push_tos_nonce' );
        if ( empty( $tos_nonce ) ) {
            $tos_nonce = wp_generate_password( 32, false, false );
            update_option( 'i10n_push_tos_nonce', $tos_nonce );
        }
        $tos_url = 'https://gordon168.com/i10n-push-tos/?from_site=' . rawurlencode( $current_site ) . '&tos_nonce=' . rawurlencode( $tos_nonce );
        ?>
        <div class="notice notice-info is-dismissible">
            <p><strong><?php esc_html_e( '歡迎使用 WordPress 中文補完計劃！', 'i10n-push-subscriber' ); ?></strong></p>
            <p>
                <?php esc_html_e( '將您的網站連接到 WordPress 中文補完計劃服務，以啟用自動化翻譯與訂閱功能。', 'i10n-push-subscriber' ); ?>
                <br>
                <?php /* translators: %s: current site URL. */ ?>
                <small><?php printf( esc_html__( '請注意：啟用後，您的訂閱將與本網站網址 (%s) 綁定，無法轉移至其他網站。', 'i10n-push-subscriber' ), '<strong>' . esc_html( get_site_url() ) . '</strong>' ); ?></small>
            </p>
            <div class="i10n-push-connect-actions" style="display:flex; flex-wrap:wrap; gap:8px; align-items:center;">
                <a id="i10n-push-tos-link" href="<?php echo esc_url( $tos_url ); ?>" target="_blank" class="button button-secondary">
                    <?php esc_html_e( '閱讀並同意服務條款', 'i10n-push-subscriber' ); ?>
                </a>
                <button id="i10n-push-check-tos-button" class="button button-secondary">
                    <?php esc_html_e( '我已同意，重新檢查', 'i10n-push-subscriber' ); ?>
                </button>
                <span id="i10n-push-tos-status"></span>
            </div>
            <div class="i10n-push-connect-actions" style="display:flex; flex-wrap:wrap; gap:8px; align-items:center; margin-top:8px;">
                <button id="i10n-push-connect-button" class="button button-primary" disabled>
                    <?php esc_html_e( '一鍵連接', 'i10n-push-subscriber' ); ?>
                </button>
                <span style="color:#6b7280;"><?php esc_html_e( '請先閱讀並同意服務條款後，才可連接網站。', 'i10n-push-subscriber' ); ?></span>
            </div>
        </div>
        <?php
    }

    public function handle_force_update_translations_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        // Force update all subscribed items.
        $this->updater->check_for_updates( null, true );

        wp_send_json_success( array( 'message' => __( '強制更新翻譯已觸發。請稍候，翻譯檔案將在背景下載並安裝。', 'i10n-push-subscriber' ) ) );
    }

    /**
     * Handle reapply request from revoked sites.
     */
    public function handle_reapply_site_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        $reason = isset( $_POST['reason'] ) ? sanitize_textarea_field( wp_unslash( $_POST['reason'] ) ) : '';
        $api_url = trailingslashit( I10N_PUSH_API_URL ) . 'reapply-site';

        $response = wp_remote_post( $api_url, array(
            'timeout' => 20,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'body' => wp_json_encode( array(
                'site_url' => get_site_url(),
                'reason' => $reason,
                'admin_email' => get_option( 'admin_email' ),
            ) ),
        ) );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => $response->get_error_message() ), 500 );
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code >= 200 && $code < 300 ) {
            // Mark locally as reapply_pending to align UI
            update_option( 'i10n_push_connection_status', 'reapply_pending' );
            wp_send_json_success( array( 'message' => $body['message'] ?? __( '已提交重新申請，請等待審核。', 'i10n-push-subscriber' ) ) );
        }

        $msg = isset( $body['message'] ) ? $body['message'] : __( '提交失敗，請稍後再試。', 'i10n-push-subscriber' );
        wp_send_json_error( array( 'message' => $msg ), $code );
    }

    /**
     * Displays a notice when the site connection is awaiting email verification.
     *
     * @since 1.0.0
     */
    public function display_pending_verification_notice() {
        // Only show to admins on admin pages, and only when the status is 'pending'.
        if ( get_option( 'i10n_push_connection_status' ) !== 'pending' || ! current_user_can( 'manage_options' ) ) {
            return;
        }
        // Show only after a real connect attempt triggered the verification email.
        if ( ! get_option( 'i10n_push_pending_verification_at' ) ) {
            return;
        }
        ?>
        <div class="notice notice-warning">
            <p>
                <strong><?php esc_html_e( 'WordPress 中文補完計劃：等待郵件驗證', 'i10n-push-subscriber' ); ?></strong>
            </p>
            <p>
                <?php printf(
                    /* translators: %s: admin email address */
                    esc_html__( '我們已發送一封驗證郵件至您的管理員信箱 (%s)。請點擊信中的連結以啟用服務。', 'i10n-push-subscriber' ),
                    '<strong>' . esc_html( get_option( 'admin_email' ) ) . '</strong>'
                ); ?>
            </p>
        </div>
        <?php
    }

    public function handle_check_tos_consent_ajax() {
        check_ajax_referer( 'i10n_push_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( '權限不足。', 'i10n-push-subscriber' ) ), 403 );
        }

        $tos_nonce = get_option( 'i10n_push_tos_nonce' );
        if ( empty( $tos_nonce ) ) {
            wp_send_json_success( array( 'consented' => false ) );
        }

        $api_url = trailingslashit( I10N_PUSH_API_URL ) . 'tos/consent-status';
        $response = wp_remote_get(
            add_query_arg(
                array(
                    'site_url'  => get_site_url(),
                    'tos_nonce' => $tos_nonce,
                ),
                $api_url
            ),
            array( 'timeout' => 20 )
        );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => $response->get_error_message() ), 500 );
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        wp_send_json_success( array(
            'consented' => ! empty( $body['consented'] ),
            'consented_at' => $body['consented_at'] ?? null,
        ) );
    }

}
