<?php
class I10n_Push_Subscriber_Downloader {

    private $plugin_name;
    private $version;

    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Downloads and installs translation files from the provided URLs.
     *
     * @param array $updates An array of update items, each with text_domain, type, new_version, package (URL), and new_translation_timestamp.
     */
    public function download_and_install( $updates ) {
        if ( empty( $updates ) ) {
            i10n_push_log( '[i10n-push-subscriber-downloader] No updates to process.' );
            return;
        }

        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        global $wp_filesystem;

        foreach ( $updates as $update ) {
            $text_domain = $update['text_domain'];
            $type = $update['type'];
            $package_url = $update['package'];
            $new_translation_timestamp = $update['new_translation_timestamp'];
            $locale = get_locale(); // zh_TW

            i10n_push_log( "[i10n-push-subscriber-downloader] Processing update for '{$text_domain}' ({$type}). Package URL: {$package_url}" );

            // Determine the base language directory (custom path to avoid official overrides)
            $lang_base_dir = ( 'plugin' === $type ) ? I10N_PUSH_CUSTOM_LANG_DIR . '/plugins' : I10N_PUSH_CUSTOM_LANG_DIR . '/themes';
            if ( ! $wp_filesystem->exists( $lang_base_dir ) ) {
                wp_mkdir_p( $lang_base_dir );
            }

            // Create a temporary file to download the package
            $temp_file = download_url( $package_url );

            if ( is_wp_error( $temp_file ) ) {
                i10n_push_log( '[i10n-push-subscriber-downloader] Failed to download package for ' . $text_domain . ': ' . $temp_file->get_error_message() );
                continue;
            }
            
            // --- Start Cleanup of old translation files for this text_domain ---
            $this->clean_old_translation_files( $text_domain, $type, $wp_filesystem );
            // --- End Cleanup ---

            $file_extension = strtolower( pathinfo( $temp_file, PATHINFO_EXTENSION ) );
            $target_file_base = $lang_base_dir . '/' . $text_domain . '-' . $locale; // e.g., /wp-content/languages/plugins/elementor-zh_TW

            if ( 'zip' === $file_extension ) {
                // Handle ZIP file: Unzip to a temporary directory, then move contents
                $unzip_dir = WP_CONTENT_DIR . '/uploads/i10n-push-temp-' . uniqid();
                if ( ! $wp_filesystem->mkdir( $unzip_dir ) ) {
                    i10n_push_log( '[i10n-push-subscriber-downloader] Failed to create unzip directory: ' . $unzip_dir );
                    @unlink( $temp_file );
                    continue;
                }

                $unzip_result = unzip_file( $temp_file, $unzip_dir );
                @unlink( $temp_file ); // Delete the temporary zip file

                if ( is_wp_error( $unzip_result ) ) {
                    i10n_push_log( '[i10n-push-subscriber-downloader] Failed to unzip package for ' . $text_domain . ': ' . $unzip_result->get_error_message() );
                    $wp_filesystem->delete( $unzip_dir, true ); // Clean up temp dir
                    continue;
                }

                $this->move_translation_files_from_dir( $unzip_dir, $lang_base_dir, $wp_filesystem );
                $wp_filesystem->delete( $unzip_dir, true ); // Clean up the unzip directory

            } elseif ( 'mo' === $file_extension ) {
                // Handle single MO file
                $destination_path = $target_file_base . '.mo';
                if ( $wp_filesystem->move( $temp_file, $destination_path, true ) ) {
                    i10n_push_log( "[i10n-push-subscriber-downloader] Moved '{$text_domain}-{$locale}.mo' to '{$lang_base_dir}'." );
                } else {
                    i10n_push_log( '[i10n-push-subscriber-downloader] Failed to move .mo file for ' . $text_domain . ' to ' . $destination_path . '.' );
                    @unlink( $temp_file );
                    continue;
                }

            } else {
                // Some hosts deliver ZIPs without a .zip suffix. Try unzip as a fallback.
                $unzip_dir = WP_CONTENT_DIR . '/uploads/i10n-push-temp-' . uniqid();
                if ( $wp_filesystem->mkdir( $unzip_dir ) ) {
                    $unzip_result = unzip_file( $temp_file, $unzip_dir );
                    @unlink( $temp_file );
                    if ( is_wp_error( $unzip_result ) ) {
                        i10n_push_log( '[i10n-push-subscriber-downloader] Unknown file type for ' . $text_domain . ': ' . $file_extension );
                        $wp_filesystem->delete( $unzip_dir, true );
                        continue;
                    }
                    $this->move_translation_files_from_dir( $unzip_dir, $lang_base_dir, $wp_filesystem );
                    $wp_filesystem->delete( $unzip_dir, true );
                } else {
                    i10n_push_log( '[i10n-push-subscriber-downloader] Failed to create unzip directory: ' . $unzip_dir );
                    @unlink( $temp_file );
                    continue;
                }
            }

            // Update the translation timestamp after successful installation
            update_option( 'i10n_push_translation_timestamp_' . $text_domain, $new_translation_timestamp );
            $timestamp_keys = get_option( 'i10n_push_translation_timestamp_keys', array() );
            $timestamp_keys[] = 'i10n_push_translation_timestamp_' . $text_domain;
            $timestamp_keys = array_values( array_unique( $timestamp_keys ) );
            update_option( 'i10n_push_translation_timestamp_keys', $timestamp_keys );
            i10n_push_log( "[i10n-push-subscriber-downloader] Updated translation timestamp for '{$text_domain}' to {$new_translation_timestamp}." );

            // Clear relevant caches to ensure new translations are loaded
            if ( function_exists( 'wp_cache_flush' ) ) {
                wp_cache_flush();
            }
            // For persistent object caches, might need more specific invalidation.
            // Example: LiteSpeed Cache (if installed)
            if ( class_exists( 'LiteSpeed_Cache' ) && method_exists( 'LiteSpeed_Cache', 'purge_all' ) ) {
                LiteSpeed_Cache::purge_all();
            }
            // Example: WP Super Cache (if installed)
            if ( function_exists( 'wp_cache_clear_cache' ) ) {
                wp_cache_clear_cache();
            }
        }
    }

    /**
     * Cleans up old translation files (including potential PHP cache files) for a given text domain.
     *
     * @param string      $text_domain  The text domain of the plugin/theme.
     * @param string      $type         'plugin' or 'theme'.
     * @param WP_Filesystem_Base $wp_filesystem The WP_Filesystem instance.
     */
    private function clean_old_translation_files( $text_domain, $type, $wp_filesystem ) {
        $locale = get_locale();
        $lang_base_dir = ( 'plugin' === $type ) ? I10N_PUSH_CUSTOM_LANG_DIR . '/plugins' : I10N_PUSH_CUSTOM_LANG_DIR . '/themes';

        // Define patterns for files to clean, including common PHP cache files
        // We target files that contain the text domain and locale.
        $patterns = array(
            $text_domain . '-' . $locale . '.mo',    // Old .mo file
            $text_domain . '-' . $locale . '.po',    // Old .po file (less common in languages dir, but possible)
            $text_domain . '-' . $locale . '.json',   // Old .json translation files (for block themes/plugins)
            $text_domain . '-' . $locale . '.l10n.php', // Cached optimized translations (e.g., elementor-zh_TW.l10n.php)
            $text_domain . '-*'. $locale . '.php',   // General PHP cache files potentially containing locale
            $text_domain . '*'. $locale . '.php',   // More general PHP cache files
            $text_domain . '-' . $locale . '*.php', // PHP files starting with domain-locale
            '*' . $text_domain . '-' . $locale . '.php', // PHP files ending with domain-locale
            '*' . $text_domain . '*'. $locale . '.php', // PHP files containing domain and locale
            // Targeting specific known patterns from cache plugins or optimized MO files
            'MO_' . $text_domain . '-' . $locale . '.php',
            'MO_' . $text_domain . '-' . $locale . '*.php',
        );

        i10n_push_log( "[i10n-push-subscriber-downloader] Cleaning old translation files for '{$text_domain}' in '{$lang_base_dir}'." );

        $dir_contents = $wp_filesystem->dirlist( $lang_base_dir );

        if ( ! empty( $dir_contents ) ) {
            foreach ( $dir_contents as $file_name => $file_data ) {
                foreach ( $patterns as $pattern ) {
                    // Convert glob pattern to regex for more robust matching
                    $regex_pattern = '/^' . str_replace( array( '.', '*', '?' ), array( '\.', '.*', '.?' ), $pattern ) . '$/i';
                    if ( preg_match( $regex_pattern, $file_name ) ) {
                        $file_path_to_delete = $lang_base_dir . '/' . $file_name;
                        if ( $wp_filesystem->delete( $file_path_to_delete ) ) {
                            i10n_push_log( "[i10n-push-subscriber-downloader] Deleted old translation file: '{$file_path_to_delete}'." );
                        } else {
                            i10n_push_log( "[i10n-push-subscriber-downloader] Failed to delete old translation file: '{$file_path_to_delete}'." );
                        }
                        break; // Move to next file after first match
                    }
                }
            }
        }
    }

    /**
     * Move .mo/.json files from a directory (recursive) into the language directory.
     *
     * @param string              $source_dir
     * @param string              $target_dir
     * @param WP_Filesystem_Base  $wp_filesystem
     */
    private function move_translation_files_from_dir( $source_dir, $target_dir, $wp_filesystem ) {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $source_dir, FilesystemIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ( $iterator as $file ) {
            if ( ! $file->isFile() ) {
                continue;
            }
            $file_ext = strtolower( pathinfo( $file->getFilename(), PATHINFO_EXTENSION ) );
            if ( ! in_array( $file_ext, array( 'mo', 'json' ), true ) ) {
                continue;
            }
            $source_path = $file->getPathname();
            $destination_path = $target_dir . '/' . $file->getFilename();
            if ( $wp_filesystem->move( $source_path, $destination_path, true ) ) {
                i10n_push_log( "[i10n-push-subscriber-downloader] Moved '{$file->getFilename()}' to '{$target_dir}'." );
            } else {
                i10n_push_log( "[i10n-push-subscriber-downloader] Failed to move '{$file->getFilename()}' from '{$source_dir}' to '{$target_dir}'." );
            }
        }
    }
}
