<?php
/**
 * Handles overriding default translation files with custom ones.
 *
 * @link       https://gordon178.com/
 * @since      1.0.0
 *
 * @package    I10n_Push_Subscriber
 * @subpackage I10n_Push_Subscriber/includes
 */

/**
 * Handles overriding default .mo files with files downloaded from the i10n Push service.
 *
 * @package    I10n_Push_Subscriber
 * @subpackage I10n_Push_Subscriber/includes
 * @author     Gordon <gordon@gordon178.com>
 */
class I10n_Push_Subscriber_Override {

    public function __construct() {
        // No need to initialize subscribed_domains here, it will be fetched dynamically.
    }

    /**
     * Hooked to `load_textdomain_mofile`. Overrides the .mo file path for subscribed text domains.
     *
     * @since 1.0.0
     * @param string $mofile Path to the .mo file.
     * @param string $domain Text domain.
     * @return string Path to the new .mo file or the original file.
     */
    public function override_mo_file( $mofile, $domain ) {
        $connection_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        $api_key = get_option( 'i10n_push_api_key', '' );
        if ( 'connected' !== $connection_status || empty( $api_key ) ) {
            return $mofile;
        }
        $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );
        $free_items = get_option( 'i10n_push_free_items', array() );
        $subscribed_domains = array_unique( array_merge( array_keys( $subscribed_items ), $free_items ) );

        if ( ! in_array( $domain, $subscribed_domains, true ) ) {
            return $mofile;
        }

        $locale = get_locale();
        $custom_base = trailingslashit( I10N_PUSH_CUSTOM_LANG_DIR );
        $custom_mofile_plugin = $custom_base . 'plugins/' . $domain . '-' . $locale . '.mo';
        $custom_mofile_theme  = $custom_base . 'themes/' . $domain . '-' . $locale . '.mo';

        if ( file_exists( $custom_mofile_plugin ) ) {
            return $custom_mofile_plugin;
        }

        if ( file_exists( $custom_mofile_theme ) ) {
            return $custom_mofile_theme;
        }

        return $mofile;
    }

    /**
     * Hooked to `load_script_translation_file` to override JS translation JSON files.
     *
     * @param string $file   The translation file path.
     * @param string $handle Script handle.
     * @param string $domain Text domain.
     * @return string Path to the new JSON file or the original file.
     */
    public function override_script_translation_file( $file, $handle, $domain ) {
        $connection_status = get_option( 'i10n_push_connection_status', 'disconnected' );
        $api_key = get_option( 'i10n_push_api_key', '' );
        if ( 'connected' !== $connection_status || empty( $api_key ) ) {
            return $file;
        }
        $subscribed_items   = get_option( 'i10n_push_subscribed_items', array() );
        $free_items = get_option( 'i10n_push_free_items', array() );
        $subscribed_domains = array_unique( array_merge( array_keys( $subscribed_items ), $free_items ) );

        if ( ! in_array( $domain, $subscribed_domains, true ) ) {
            return $file;
        }

        $locale      = determine_locale();
        $custom_base = trailingslashit( I10N_PUSH_CUSTOM_LANG_DIR );
        $custom_json = $custom_base . 'plugins/' . $domain . '-' . $locale . '.json';

        if ( file_exists( $custom_json ) ) {
            return $custom_json;
        }

        // Themes can also enqueue scripts; fall back to themes path if present.
        $custom_json_theme = $custom_base . 'themes/' . $domain . '-' . $locale . '.json';
        if ( file_exists( $custom_json_theme ) ) {
            return $custom_json_theme;
        }

        return $file;
    }
}
