<?php

class I10n_Push_Subscriber_Updater {

    private $plugin_name;
    private $version;
    private $api_key;

    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->api_key = get_option( 'i10n_push_api_key', '' );
    }

    /**
     * Checks for translation updates for specified items or all subscribed items.
     * In dev mode, it sends an email with the request payload instead of making an API call.
     *
     * @param array|null $items_to_check Optional. An array of text domains to check. If null, checks all subscribed items.
     * @return array The API response, or an empty array on failure/dev mode.
     */
    public function check_for_updates( $items_to_check = null, $force_update = false ) {
        i10n_push_log( '[i10n-push-subscriber] check_for_updates called.' . ($force_update ? ' (Force Update)' : '') );

        // If API key is missing, avoid calling API to prevent false revokes
        if ( empty( $this->api_key ) ) {
            i10n_push_log( '[i10n-push-subscriber] Skipping update check: API key is empty.' );
            return [];
        }

        // --- Caching Logic ---
        $transient_key = 'i10n_push_update_check_' . md5( json_encode( $items_to_check ) );
        if ( ! $force_update ) {
            $cached_data = get_transient( $transient_key );
            if ( false !== $cached_data && isset( $cached_data['response'] ) && isset( $cached_data['items_data'] ) ) {
                i10n_push_log( '[i10n-push-subscriber] Update check response found in cache.' );
                // We still need to process downloads if any updates were cached.
                $this->process_updates_from_response( $cached_data['response'], $cached_data['items_data'] );
                return $cached_data['response'];
            }
        }
        // --- End Caching Logic ---

        $items_data = [];
        $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );
        $free_items = get_option( 'i10n_push_free_items', array() );

        if ( empty( $subscribed_items ) && empty( $free_items ) ) {
            i10n_push_log( '[i10n-push-subscriber] No subscribed or free items found.' );
        }

        // Determine which items to check
        if ( ! is_null( $items_to_check ) && is_array( $items_to_check ) ) {
            $domains_to_process = $items_to_check;
            i10n_push_log( '[i10n-push-subscriber] Checking specific items: ' . implode( ', ', $domains_to_process ) );
        } else {
            $domains_to_process = array_unique( array_merge( array_keys( $subscribed_items ), $free_items ) );
            i10n_push_log( '[i10n-push-subscriber] Checking all subscribed/free items: ' . implode( ', ', $domains_to_process ) );
        }

        // Collect metadata for each item
        foreach ( $domains_to_process as $domain ) {
            $is_free = in_array( $domain, $free_items, true );
            // Ensure the item is actually subscribed and active unless it's free tier
            if ( ! $is_free && ( ! isset( $subscribed_items[ $domain ] ) || $subscribed_items[ $domain ]['status'] !== 'active' ) ) {
                i10n_push_log( "[i10n-push-subscriber] Item {$domain} is skipped. Status: " . ( $subscribed_items[ $domain ]['status'] ?? 'not subscribed' ) );
                continue;
            }

            $item_type = 'unknown';
            $item_version = '0.0';
            $translation_timestamp = $force_update ? 0 : get_option( 'i10n_push_translation_timestamp_' . $domain, 0 );

            // Try to get plugin data
            if ( ! function_exists( 'get_plugins' ) ) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $plugins = get_plugins();
            foreach ( $plugins as $plugin_path => $plugin_data ) {
                $plugin_domain = dirname( $plugin_path );
                if ( '.' === $plugin_domain ) {
                    $plugin_domain = preg_replace( '/\.(php)$/', '', basename( $plugin_path ) );
                }
                if ( $plugin_domain === $domain ) {
                    $item_type = 'plugin';
                    $item_version = $plugin_data['Version'];
                    break;
                }
            }

            // If not a plugin, try to get theme data
            if ( $item_type === 'unknown' ) {
                $theme = wp_get_theme( $domain );
                if ( $theme->exists() ) {
                    $item_type = 'theme';
                    $item_version = $theme->get( 'Version' );
                }
            }
            
            i10n_push_log( "[i10n-push-subscriber] Found item {$domain}: type={$item_type}, version={$item_version}" );

            $items_data[] = [
                'text_domain'           => $domain,
                'type'                  => $item_type,
                'version'               => $item_version,
                'translation_timestamp' => $translation_timestamp,
            ];
        }

        if ( empty( $items_data ) ) {
            i10n_push_log( '[i10n-push-subscriber] No valid items to check after processing.' );
            return [];
        }

        $request_payload = [
            'site_url' => get_site_url(),
            'items'    => $items_data,
        ];
        
        // Add force_update parameter if true
        if ( $force_update ) {
            $request_payload['force_update'] = true;
        }
        
        i10n_push_log( '[i10n-push-subscriber] Sending API request to ' . I10N_PUSH_API_URL . '/check-updates' );

        // In dev mode, send an email instead of making an API call
        if ( defined( 'I10N_PUSH_DEV_MODE' ) && I10N_PUSH_DEV_MODE ) {
            $this->send_test_email(
                '更新檢查請求 (測試模式)',
                '模擬發送到伺服器的更新檢查請求內容',
                [ '請求內容' => '<pre>' . json_encode( $request_payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE ) . '</pre>' ]
            );
            
            // --- Dynamically Simulate a server response for testing the Downloader ---
            $simulated_updates = [];
            foreach ( $request_payload['items'] as $item ) {
                $simulated_updates[] = [
                    'text_domain' => $item['text_domain'],
                    'type'        => $item['type'], // Add type for downloader
                    'new_version' => $item['version'],
                    'package' => 'https://example.com/download/' . $item['text_domain'] . '-' . $item['version'] . '-zh_TW.mo?token=...&expires=...&signature=...',
                    'new_translation_timestamp' => time(),
                ];
            }
            
            if ( ! empty( $simulated_updates ) ) {
                $downloader = new I10n_Push_Subscriber_Downloader( $this->plugin_name, $this->version );
                $downloader->download_and_install( $simulated_updates );
            }
            
            return []; // No actual API response in dev mode
        }

        // --- Production API Call ---
        $api_url = I10N_PUSH_API_URL . '/check-updates';
        $response = wp_remote_post(
            $api_url,
            [
                'timeout' => 30,
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->api_key,
                    'Content-Type'  => 'application/json',
                    'Accept'        => 'application/json',
                    'Referer'       => get_site_url(),
                ],
                'body'    => wp_json_encode( $request_payload ),
                'user-agent' => 'i10n-push-subscriber/' . $this->version . '; ' . get_site_url(),
            ]
        );

        if ( is_wp_error( $response ) ) {
            i10n_push_log( '[i10n-push-subscriber] API Error: ' . $response->get_error_message() );
            return [];
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        if ( 200 !== $response_code ) {
            $response_body = wp_remote_retrieve_body( $response );
            if ( in_array( $response_code, array( 401, 403 ), true ) ) {
                $current_status = get_option( 'i10n_push_connection_status', 'disconnected' );
                // 只有在已有 key 且目前狀態是 connected 時，才強制標記 revoked
                if ( ! empty( $this->api_key ) && 'connected' === $current_status ) {
                    update_option( 'i10n_push_connection_status', 'revoked' );
                    delete_option( 'i10n_push_api_key' );
                }
            }
            i10n_push_log( "[i10n-push-subscriber] API returned non-200 status: {$response_code}. Body: {$response_body}" );
            return [];
        }

        $response_data = json_decode( wp_remote_retrieve_body( $response ), true );

        // Cache the response and the item data together for 1 hour
        $cache_data = [
            'response'   => $response_data,
            'items_data' => $items_data,
        ];
        set_transient( $transient_key, $cache_data, HOUR_IN_SECONDS );
        $update_check_keys = get_option( 'i10n_push_update_check_keys', array() );
        $update_check_keys[] = $transient_key;
        $update_check_keys = array_values( array_unique( $update_check_keys ) );
        update_option( 'i10n_push_update_check_keys', $update_check_keys );
        
        i10n_push_log( '[i10n-push-subscriber] API Response: ' . json_encode( $response_data ) );

        $this->process_updates_from_response( $response_data, $items_data );

        return $response_data;
    }

    /**
     * Processes the 'updates' array from an API response and initiates downloads.
     *
     * @param array $response_data The full API response data.
     * @param array $items_data    The original items data sent in the request, used to map types.
     */
    private function process_updates_from_response( $response_data, $items_data ) {
        if ( isset( $response_data['updates'] ) && ! empty( $response_data['updates'] ) ) {
            i10n_push_log( '[i-10n-push-subscriber] Updates found. Starting download process.' );

            // Create a map of text_domain => type from the original request data.
            $type_map = array_column( $items_data, 'type', 'text_domain' );

            // Add the 'type' back into the updates array from the server response.
            $updates_with_type = array_map(
                function ( $update ) use ( $type_map ) {
                    if ( isset( $type_map[ $update['text_domain'] ] ) ) {
                        $update['type'] = $type_map[ $update['text_domain'] ];
                    }
                    return $update;
                },
                $response_data['updates']
            );

            // Skip downloads when local translation timestamp is already up-to-date.
            $updates_with_type = array_filter(
                $updates_with_type,
                function ( $update ) {
                    $domain = $update['text_domain'] ?? '';
                    $server_ts = isset( $update['new_translation_timestamp'] ) ? (int) $update['new_translation_timestamp'] : 0;
                    if ( empty( $domain ) || $server_ts <= 0 ) {
                        return true; // If data is incomplete, proceed to be safe.
                    }
                    $local_ts = (int) get_option( 'i10n_push_translation_timestamp_' . $domain, 0 );

                    // If the custom file is missing, force download even when timestamps match.
                    $locale = get_locale();
                    $type   = $update['type'] ?? 'plugin';
                    $base   = ( 'plugin' === $type ) ? I10N_PUSH_CUSTOM_LANG_DIR . '/plugins' : I10N_PUSH_CUSTOM_LANG_DIR . '/themes';
                    $target_file = trailingslashit( $base ) . $domain . '-' . $locale . '.mo';
                    $has_file = file_exists( $target_file );

                    // Allow small clock skew but only skip when file已存在且時間戳實質相同.
                    $skew_tolerance = 10 * MINUTE_IN_SECONDS;
                    if ( $has_file && $local_ts >= $server_ts && ( $local_ts - $server_ts ) <= $skew_tolerance ) {
                        i10n_push_log( "[i10n-push-subscriber] Skipping download for {$domain}; local timestamp {$local_ts} within tolerance of server {$server_ts} and file exists." );
                        return false; // Skip only when effectively equal (within tolerance) AND file present.
                    }
                    return true;
                }
            );

            $downloader = new I10n_Push_Subscriber_Downloader( $this->plugin_name, $this->version );
            $downloader->download_and_install( $updates_with_type );
        } else {
            i10n_push_log( '[i10n-push-subscriber] No updates found in server response.' );
        }
    }

    /**
     * Handles the `upgrader_process_complete` hook to trigger update checks after plugin/theme upgrades.
     *
     * @param WP_Upgrader $upgrader_object The WP_Upgrader instance.
     * @param array       $options         Array of bulk item update data.
     */
    public function on_upgrade( $upgrader_object, $options ) {
        // Only proceed for plugin or theme updates
        if ( ! in_array( $options['action'], ['update', 'install'], true ) || ! in_array( $options['type'], ['plugin', 'theme'], true ) ) {
            return;
        }

        $items_to_check = [];

        if ( 'plugin' === $options['type'] ) {
            if ( isset( $options['plugins'] ) ) {
                foreach ( $options['plugins'] as $plugin_path ) {
                    $domain = dirname( $plugin_path );
                    if ( '.' === $domain ) {
                        $domain = preg_replace( '/\.(php)$/', '', basename( $plugin_path ) );
                    }
                    $items_to_check[] = $domain;
                }
            }
        } elseif ( 'theme' === $options['type'] ) {
            if ( isset( $options['themes'] ) ) {
                foreach ( $options['themes'] as $theme_slug ) {
                    $items_to_check[] = $theme_slug;
                }
            }
        }

        // Optimization: Filter out items that are NOT subscribed locally before triggering the check.
        $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );
        $free_items = get_option( 'i10n_push_free_items', array() );
        $valid_items_to_check = [];

        foreach ( $items_to_check as $domain ) {
            // Only proceed if the item is in our local subscription list.
            // The check_for_updates method will further verify if the subscription is 'active'.
            if ( array_key_exists( $domain, $subscribed_items ) || in_array( $domain, $free_items, true ) ) {
                $valid_items_to_check[] = $domain;
            }
        }

        if ( ! empty( $valid_items_to_check ) ) {
            // Trigger update check only for subscribed items
            $this->check_for_updates( $valid_items_to_check );
        }
    }

    /**
     * Handles the `activated_plugin` hook to trigger an update check upon plugin reactivation.
     *
     * @param string $plugin_path Path to the plugin file relative to the plugins directory.
     */
    public function on_plugin_reactivation( $plugin_path ) {
        $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );
        $free_items = get_option( 'i10n_push_free_items', array() );
        $domain = dirname( $plugin_path );
        if ( '.' === $domain ) {
            $domain = preg_replace( '/\.(php)$/', '', basename( $plugin_path ) );
        }

        // If the reactivated plugin is in our subscription list, force a check for it.
        if ( array_key_exists( $domain, $subscribed_items ) || in_array( $domain, $free_items, true ) ) {
            i10n_push_log( "[i10n-push-subscriber] Reactivated plugin '{$domain}' is subscribed. Forcing translation update check." );
            $this->check_for_updates( array( $domain ), true );
        }
    }

    /**
     * Handles the `switch_theme` hook to trigger an update check when a theme is activated.
     *
     * @param string    $new_name  Name of the new theme.
     * @param WP_Theme  $new_theme WP_Theme instance of the new theme.
     */
    public function on_theme_switch( $new_name, $new_theme ) {
        $subscribed_items = get_option( 'i10n_push_subscribed_items', array() );
        $free_items = get_option( 'i10n_push_free_items', array() );
        $domain = $new_theme->get_stylesheet(); // The theme's text domain is its slug

        // If the newly activated theme is in our subscription list, force a check for it.
        if ( array_key_exists( $domain, $subscribed_items ) || in_array( $domain, $free_items, true ) ) {
            i10n_push_log( "[i10n-push-subscriber] Switched to subscribed theme '{$domain}'. Forcing translation update check." );
            $this->check_for_updates( array( $domain ), true );
        }
    }

    /**
     * Schedules the daily update check via WP-Cron.
     */
    public function schedule_daily_check() {
        if ( ! wp_next_scheduled( 'i10n_push_daily_update_check' ) ) {
            wp_schedule_event( time(), 'daily', 'i10n_push_daily_update_check' );
        }
    }

    /**
     * The callback for the daily WP-Cron event.
     */
    public function daily_update_check_callback() {
        // Check for updates for all subscribed items
        $this->check_for_updates();
    }

    /**
     * Sends a test email in development mode.
     *
     * @param string $subject The email subject.
     * @param string $title   The email title.
     * @param array  $data    The data to include in the email body.
     */
    private function send_test_email( $subject, $title, $data ) {
        $to = 'i10n@gordon168.com';
        $subject = '[i10n Push Test] ' . $subject . ' - ' . get_site_url();
        $body = "<h1>{$title}</h1>";
        $body .= '<p><strong>網站網址:</strong> ' . get_site_url() . '</p>';
        $body .= '<p><strong>管理員 Email:</strong> ' . get_option( 'admin_email' ) . '</p>';
        
        foreach ( $data as $key => $value ) {
            $body .= '<p><strong>' . esc_html( $key ) . ':</strong></p>';
            if ( is_array( $value ) && ! empty( $value ) ) {
                $body .= '<ul>';
                foreach ( $value as $item ) {
                    $body .= '<li>' . esc_html( $item ) . '</li>';
                }
                $body .= '</ul>';
            } elseif ( ! is_array( $value ) ) {
                $body .= '<p>' . esc_html( $value ) . '</p>';
            } else {
                $body .= '<p>N/A</p>';
            }
        }

        $headers = array( 'Content-Type: text/html; charset=UTF-8' );
        wp_mail( $to, $subject, $body, $headers );
    }
}
