<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://gordon178.com/
 * @since      1.0.0
 *
 * @package    I10n_Push_Subscriber
 * @subpackage I10n_Push_Subscriber/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    I10n_Push_Subscriber
 * @subpackage I10n_Push_Subscriber/includes
 * @author     Gordon <gordon@gordon178.com>
 */
class I10n_Push_Subscriber {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      I10n_Push_Subscriber_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'I10N_PUSH_SUBSCRIBER_VERSION' ) ) {
            $this->version = I10N_PUSH_SUBSCRIBER_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'i10n-push-subscriber';

        $this->load_dependencies();
        $this->set_locale();
        $this->initialize_modules();
    }

    /**
     * Initialize plugin modules and define their hooks.
     *
     * @since    1.0.0
     * @access   private
     */
    private function initialize_modules() {
        $plugin_updater = new I10n_Push_Subscriber_Updater( $this->get_plugin_name(), $this->get_version() );
        $plugin_admin = new I10n_Push_Subscriber_Admin( $this->get_plugin_name(), $this->get_version(), $plugin_updater );
        $plugin_override = new I10n_Push_Subscriber_Override();

        // Define hooks for admin, public, updater, and override modules
        $this->define_admin_hooks( $plugin_admin );
        $this->define_public_hooks(); // No public hooks yet
        $this->define_updater_hooks( $plugin_updater );
        $this->define_override_hooks( $plugin_override );
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - I10n_Push_Subscriber_Loader. Orchestrates the hooks of the plugin.
     * - I10n_Push_Subscriber_i18n. Defines internationalization functionality.
     * - I10n_Push_Subscriber_Admin. Defines all hooks for the admin area.
     * - I10n_Push_Subscriber_Updater. Defines all hooks for the updater functionality.
     * - I10n_Push_Subscriber_Downloader. Handles downloading of translation files.
     * - I10n_Push_Subscriber_Override. Handles overriding of translation files.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-loader.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-i18n.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-admin.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-updater.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-downloader.php';
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-override.php';
        // require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-i10n-push-subscriber-public.php';

        $this->loader = new I10n_Push_Subscriber_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the I10n_Push_Subscriber_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new I10n_Push_Subscriber_i18n();
        $this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks( $plugin_admin ) {
        $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
        $this->loader->add_action( 'admin_notices', $plugin_admin, 'display_connect_notice' );
        $this->loader->add_action( 'admin_notices', $plugin_admin, 'display_pending_verification_notice' );
        $this->loader->add_action( 'admin_notices', $plugin_admin, 'display_smart_notification' );
        $this->loader->add_action( 'wp_ajax_i10n_push_connect', $plugin_admin, 'handle_connect_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_check_verification', $plugin_admin, 'handle_check_verification_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_save_api_key', $plugin_admin, 'handle_save_api_key_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_create_order', $plugin_admin, 'handle_create_order_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_request_quote', $plugin_admin, 'handle_request_quote_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_cancel_all_subscriptions', $plugin_admin, 'handle_cancel_all_subscriptions_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_dismiss_notice', $plugin_admin, 'handle_dismiss_notice_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_get_available_items', $plugin_admin, 'handle_get_available_items_ajax' );
        $this->loader->add_action( 'wp_ajax_i10n_push_force_refresh', $plugin_admin, 'handle_force_refresh_ajax' );
        $this->loader->add_action( 'admin_init', $plugin_admin, 'maybe_refresh_available_items' );
        $this->loader->add_action( 'admin_init', $plugin_admin, 'maybe_trigger_activation_update' );

        // Hooks to clear the available items cache when plugins/themes are changed
        $this->loader->add_action( 'upgrader_process_complete', $this, 'clear_items_cache' );
        $this->loader->add_action( 'deleted_plugin', $this, 'clear_items_cache' );
        $this->loader->add_action( 'activated_plugin', $plugin_admin, 'mark_items_refresh_needed' );
        $this->loader->add_action( 'deactivated_plugin', $plugin_admin, 'mark_items_refresh_needed' );
        $this->loader->add_action( 'switch_theme', $plugin_admin, 'mark_items_refresh_needed' );
        $this->loader->add_action( 'delete_theme', $this, 'clear_items_cache' );
    }

    /**
     * Clears the cached list of available translation items.
     *
     * @since    1.0.0
     */
    public function clear_items_cache() {
        delete_transient( 'i10n_push_available_items' );
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        // $plugin_public = new I10n_Push_Subscriber_Public( $this->get_plugin_name(), $this->get_version() );
        // Hooks for the public side will be registered here.
    }

    /**
     * Register all of the hooks related to the updater functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_updater_hooks( $plugin_updater ) {
        $this->loader->add_action( 'upgrader_process_complete', $plugin_updater, 'on_upgrade', 10, 2 );
        $this->loader->add_action( 'activated_plugin', $plugin_updater, 'on_plugin_reactivation', 10, 1 );
        $this->loader->add_action( 'switch_theme', $plugin_updater, 'on_theme_switch', 10, 2 );

        // Cron job for periodic checks
        $cron_hook = 'i10n_push_daily_update_check';
        $this->loader->add_action( $cron_hook, $plugin_updater, 'daily_update_check_callback' );

        // Schedule the event if it's not already scheduled
        // This will be called on plugin activation as well
        $plugin_updater->schedule_daily_check();
    }

    /**
     * Register all of the hooks related to the override functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_override_hooks( $plugin_override ) {
        $this->loader->add_filter( 'load_textdomain_mofile', $plugin_override, 'override_mo_file', 10, 2 );
        $this->loader->add_filter( 'load_script_translation_file', $plugin_override, 'override_script_translation_file', 10, 3 );
    }

    /**
     * Adds custom cron schedules.
     *
     * @since 1.0.0
     * @param array $schedules An array of non-default cron schedules.
     * @return array Filtered array of non-default cron schedules.
     */
    // This method is no longer needed as Updater handles its own scheduling
    // public function add_cron_schedules( $schedules ) {
    //     $schedules['twice_daily'] = array(
    //         'interval' => 43200, // 12 hours in seconds
    //         'display'  => __( 'Twice Daily' ),
    //     );
    //     return $schedules;
    // }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    I10n_Push_Subscriber_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
}
